; RECORD.ASM - Returns the 32-bit address of a table containing mode status
;              information in DX:AX. The status information represents a
;	       complete accounting of the operation of the Hercules graphics
;	       and RamFont modes. This routine is vectored in INT 11h.
;	       If the entry conditions are not appropriate, the normal INT 11h
;	       result is returned in AX. A resident reset routine is also
;	       available which will initialize the record and the device to
;	       the power-on state.

; * * * *    Entry Condition :

; To gain access to the Mode Record, an "illegal" flags register value
; must have been placed on the stack by the INT 11h instruction. We
; define an illegal flags value as one in which the sign flag and the zero
; flag are set (a zero is not negative to the CPU), and the parity flag is
; low (zero does not imply an odd number of 1's). Note that bit 1 of the flags
; register is unused, but always held high. The entry code for the address
; of the mode record is C0h.

; The Mode Record is installed by the HGC program. The Hercules InColor
; utilities HGC.COM, SETCOLOR.COM, PALETTE.COM, and RAMFONT.COM, all
; update the Mode Record, if present. The Hercules supplied InColor Card
; drivers for 1-2-3, Symphony, and AutoCad also maintain the Record.

; * * * *    Calling Sequence :

;	mov	ah,C0h		; entry code into AH
;	sahf			; stuff this into low byte of flags register
;	int	11h		; issue interrupt 11h

; To verify the presence of a Mode Record, look for the string 'HCT' at 
; byte offset 8 into the record. If 'HCT' is found, compare the rest of 
; the "manufacturer_id" string to make sure that the record is appropriate.
; This scheme allows a chain of resident records to be installed, each
; one accomodating a different device.

; Examine the "look_for_record" procedure in the RAMFONT.ASM file for 
; an illustration of a technique for verifying the presence of a valid
; Mode Record. Also included in the RAMFONT.ASM source listing is a useful
; record STRUC(ture).

; This file wants to become a .COM program :  MASM record;
; 					      LINK record;
;					      EXE2BIN record.exe record.com

equipment	equ	11h		; Interrupt vector number

dos_call	equ	21h		; DOS functions
terminate	equ	4c00h
terminate_res	equ	3100h
print_string	equ	9

entry_code	equ	0C0h		; code to gain access to the Record

cr		equ	13
lf		equ	10

dmc_port	equ	3b8h		; Display Mode Control Port
index_reg	equ	3b4h		; 6845 Index Register

buffer_start	equ	0b000h		; Segment address of video buffer

off_		equ	0		; INSTALL_INT macro needs these
seg_		equ	2

; install_int - takes as its argument the interrupt number to substitute with
;               our routine. Requires a (dd) declaration with the name
;		"int_num"_old_ptr for the old vector contents, and a label
;               for the new routine "int_num"_new.
;               assumes DS=CS     AX,BX,DX,ES destroyed

install_int	macro	int_num	
	mov	ax,(35h shl 8) or int_num
	int	dos_call		; ES:BX = seg:offset of old vector
	mov	word ptr int_num&_old_ptr+off_,bx
	mov	word ptr int_num&_old_ptr+seg_,es
	lea	dx,int_num&_new
	mov	ax,(25h shl 8) or int_num
	int	dos_call		; DS:DX = new_cs:new_ip
	endm

record	segment	para	public	'CODE'
	assume	cs:record, ds:record
	org	100h

start	proc	far
	jmp	loader

; * * * * * * * * * Beginning Of Record * * * * * * * * * *

record_start:

ptr_to_next	dd	0		; 32-bit address of the next record

ptr_to_reset	dd	0		; address of resident reset routine 

manufacturer_id	db	'HCT 1.GB222',0,0,0,0,0,0,0     ; 18 bytes

version_id	db	1		; this is the first version

device_type	db	1		; 1 = HGC, 2 = CGA, 3 = EGA

record_length	dw	(r_end_record - record_start) + 3

rec_config_port db	0		; copy of Configuration Port (03BF)

rec_d_m_c_port	db	08h		; copy of Display Mode Control (03B8)

rec_buffer_len	dw	2000		; size of buffer

rec_blank_char  dw	720h		; value for clearing screen

r_reserved	db	16 dup (0)	; for future use

; Copy of 6845 registers - including "psuedo" 6845 Hercules registers

r_0		db	61h		; GB102, GB112, GB222
r_1		db	50h		; GB102, GB112, GB222
r_2		db	52h		; GB102, GB112, GB222
r_3		db	0fh		; GB102, GB112, GB222
r_4		db	19h		; GB102, GB112, GB222
r_5		db	06h		; GB102, GB112, GB222
r_rows_visible	db	19h		; GB102, GB112, GB222
r_7		db	19h		; GB102, GB112, GB222
r_8		db	2		; GB102, GB112, GB222
r_height_less_1	db	0dh		; GB102, GB112, GB222
r_cursor_start	db	0bh		; GB102, GB112, GB222
r_cursor_stop	db	0ch		; GB102, GB112, GB222
r_0ch		db	0		; GB102, GB112, GB222
r_0dh		db	0		; GB102, GB112, GB222
r_cursor_pos_hi	db	0		; GB102, GB112, GB222
r_cursor_pos_lo	db	0		; GB102, GB112, GB222
r_lp_trip_hi	db	0		; GB102, GB112, GB222
r_lp_trip_low	db	0		; GB102, GB112, GB222
r_12h		db	0		; unused
r_13h		db	0		; unused
r_xModeReg	db	00h		; GB112, GB222
r_ScoreReg	db	00h		; GB112, GB222
r_StrikeReg	db	00h		; GB112, GB222
r_ExceptionReg	db	20h		; GB222
r_PlaneMaskReg	db	0fh		; GB222
r_R_W_CtrlReg   db	40h		; GB222
r_R_W_ColReg	db	0fh		; GB222
r_LatchProtect	db	0		; GB222
r_PaletteReg	db	0		; GB222
r_1dh		db	0		; unused
r_1eh		db	0		; unused
r_1fh		db	0		; unused

r_palette	db      0		; These 16 bytes hold a copy of  
		db	1		; the GB222 palette.
		db	2
		db	3
		db	4
		db	5
		db	6
		db	7
		db	8
		db	9
		db	0ah
		db	0bh
		db	0ch
		db	0dh
		db	0eh
		db	0fh

r_type_usage	db	0		; six bytes provide a map of
		db	0		; 48 "Types". Each bit represents
		db	0		; a 4K block of character storage
		db	0		; RAM. For the GB112, only the first
		db	0		; 12 bits are meaningful.
		db	0

r_reserved_1	db	32 dup (0)	; for future use

r_end_record:
		db	'END'		; end of record

; * * * * * * * * * * * End Of Record * * * * * * * * * * * *

equipment_old_ptr dd	?

equipment_new	label	far		; When installed, INT 11h points here.
	sti
	push	bp			; preserve base pointer
	mov	bp,sp			; initialize it to current stack
	
	mov	al,[bp + 6]		; get modified flags byte into AL
	and	al,0C0h			; mask the meaningful bits

	cmp	al,0C0h			; entry code for record address
	je	record_address		; request

	pop	bp			; if we are here, the entry
					; condition has not been satisfied
					; clean up stack
	jmp	do_equipment_check	; do the normal INT 11h.

record_address:
	mov	dx,cs			; if we are here, caller wants the
					; address of the Mode Record
					; Returns SEGMENT:OFFSET of Mode
	mov	ax,offset record_start	; Record in DX:AX

	pop	bp
	iret

; * * * * * * * * *  Resident Reset Routine * * * * * * * * * *
; This routine sets the installed Hercules card back to its power-on
; condition with two exceptions. 1) it does not alter the Configuration
; setting, if the board is in HALF or FULL configuration it is not set
; back to DIAG. 2) the InColor Card's Palette is not changed.
; AX is destroyed, everything else preserved.

power_on_values	db	08h,0D0h,07h,20h,07h,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
		db	61h,50h,52h,0fh,19h,06h,19h,19h,02h,0dh,0bh,0ch,0,0
		db	0,0,0,0,0,0,0,00h,00h,20h,0fh,40h,0fh,0

reset	proc	far

	push	ds
	push	es
	push	si
	push	di
	push	cx
	push	dx

	mov	ax,cs			; initialize DS and ES 
	mov	ds,ax
	mov	es,ax

	cld				; move in right direction

	mov	si,offset power_on_values  ; update record with power-on
	mov	di,offset rec_d_m_c_port   ; values
	mov	cx,49
rep	movsb

	mov	al,rec_d_m_c_port	; get record of current control port
	and	al,11110111b		; turn video-enable bit off
					; but preserve other states
	mov	dx,dmc_port
	out	dx,al			; disable video

	mov	cx,28			; initialize only first 28 registers
	mov	dx,index_reg		; point DX at 6845 index register
	xor	ah,ah			; start from register 0
	mov	si,offset r_0		; start of data table

	cli				; don't allow interrupts
parms:
	nop				; it has been noted that at 8Mhz and
	nop				; higher, some 6845's on the GB102
					; can't keep up, this pause is for them
	mov	al,ah
	out	dx,al			; output to index register
	inc	dx			; point DX to 6845 data register
	lodsb				; transfer value from SI to AL
	out	dx,al			; send data to 6845 register
	inc	ah			; point to next 6845 register
	dec	dx			; point again to index register
	loop	parms			; loop executes 27 times initializing
					; all timing and mode set registers.

	sti				; allow interrupts again

	mov	cx,rec_buffer_len	; number of words to clear
	mov	ax,buffer_start
	mov	es,ax
	xor	di,di
	mov	ax,rec_blank_char	; character used to clear buffer
rep	stosw
	mov	dx,dmc_port		; enable video
	mov	al,rec_d_m_c_port
	out	dx,al

	pop	dx
	pop	cx
	pop	di
	pop	si
	pop	es
	pop	ds

	ret

reset	endp

; * * * * * * *  End of Reset Routine * * * * * * * * * *

do_equipment_check:
	jmp	cs:equipment_old_ptr	; If neither of the above entry
					; conditions are fulfilled, then jump
					; to the normal INT 11h routine.

; * * * * * * * End of Resident Portion * * * * * * * * *

loader:
	mov	ax,cs
	mov	ds,ax

; first, see if a Record is installed by comparing the first ten bytes
; of this INT11 handler with what is currently vectored by INT 11.

	mov	ax,(35h shl 8) or equipment	; get address from vector
	int	dos_call			; table into ES:[BX].

	mov	di,bx
	lea	si,equipment_new
	mov	cx,10			; Compare 10 bytes
	cld
repz	cmpsb				; This instruction will repeat as
					; long as the zero flag remains set.
					; If ZF clears, the compare has
					; failed.
	jcxz	already_loaded		; 10 byte compare complete - we
					; must be loaded already.

	install_int	equipment	; put our code into INT 11h

	mov	word ptr ptr_to_reset,offset reset  ; initialize the record
             					    ; with the address of the
	mov	word ptr ptr_to_reset+2,cs	    ; reset routine

	jmp	quit_leaving_record

already_loaded:
; If there is already this INT11h handler then we know a Record is installed.
; Now we need to see if that record is different than ours.

	mov	ah,entry_code
	sahf
	int	11h			; get and save address

check_id:
	mov	word ptr record_addr,ax
	mov	word ptr record_addr+2,dx

	mov	es,dx			; use record address to compare
	mov	di,ax			; ID strings
	add	di,8			; add offset to manufacturer_id

	xor	cx,cx
	mov	si,offset manufacturer_id
	mov	cl,11			; eleven bytes identify our Record	

	cld
repz	cmpsb

	jcxz	loaded			; if we get here and CX is 0 then
					; our record is resident already

; if not our record, is there a pointer to another installed record ?

        les	di,record_addr 		; look at the ptr_to_next
					; field in the present record
	cmp	word ptr es:[di],0 	; if this field is 0 then no
	je	install_ours		; further records exist

	mov	ax,es:[di]		; if field contains a pointer to
	mov	dx,es:[di+2]		; another record - use that address
					; to keep checking
	jmp	check_id

install_ours:

	mov	word ptr es:[di],offset record_start  ; point to our Record
	mov	word ptr es:[di+2],cs		      ; in the installed
	  					      ; Record's ptr_to_next
						      ; field

	mov	word ptr ptr_to_reset,offset reset    ; initialize Reset
	mov	word ptr ptr_to_reset+2,cs


quit_leaving_record:

	mov	ah,print_string		; display appropriate message
	lea	dx,inst_msg_1
	int	dos_call

	mov	dx,offset loader + 15	; get byte after resident portion + 15
	shr	dx,1
	shr	dx,1
	shr	dx,1
	shr	dx,1			; divide by 16
					; DX now contains # of paragraphs
					; of memory to preserve
	mov	ax,terminate_res
	int	dos_call		; normal return with routine installed

loaded:
	mov	ah,print_string		; display already installed message
	lea	dx,inst_msg_2
	int	dos_call

	mov	ax,terminate		; normal return if already loaded
	int	dos_call

inst_msg_1	 db	cr,lf,'  The Mode Record is now installed.',cr,lf,'$'
inst_msg_2	 db	cr,lf,'  The Mode Record has already been installed.'
		 db	cr,lf,'$'

record_addr	dd	?

start	endp
record 	ends
	end	start
