/* --------------------------------- autop.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Autopilot related functions.
*/

#include <stdlib.h>
#include <string.h>

#include "fly.h"
#include "plane.h"

#define IS_CLASSIC(p)	(1 == EEP(p)->opt[0])

/*#define	DEBUG_SPEED*/

#define	RANDTIME	8000		/* in milliseconds */
#define	MAXPITCH	(D90/3*2)
#define	MAXALT		5000
#define	MINALT		500
#define	MAXRANGE	(5000L*VONE)
#define	MINCLIMB	(500*VONE)

#ifndef DEBUG_SPEED
#define	MINSPEED	(100L*VONE)
#define	MAXSPEED	(200L*VONE)
#define	MINSHOOT	(2000L*VONE)
#else
#define	MINSPEED	(300L*VONE)		/* overflow testing */
#define	MAXSPEED	(2000L*VONE)
#define	MINSHOOT	(3000L*VONE)
#endif

#define	PATROLSPEED	250
#define	RANDOMSPEED	150

#define	MINDIST		(250L*VONE)

#define SETFLAPS	30			/* flaps for takeoff */
#define F16FACT		(D90)			/* smaller = faster */
#define CLFACT		(D90/4)			/* smaller = faster */
#define SENSPITCH	(D90/16)		/* smaller = faster */
#define SENSROLL	(D90/4 )		/* smaller = faster */
#if 0
#define LIFTOFF		(2*fmul (EEP(p)->liftoff_speed, 8428))	/* meters/sec */
#else
#define LIFTOFF		(fmul (EEP(p)->liftoff_speed, 8428/2*3))	/* meters/sec */
#endif
#define ABMIN		20			/* when to set afterburner */

static void FAR
SetMeetCorrection (OBJECT *p, OBJECT *target, VECT R)
{
	int	dist, rv, dt, speed;
	LVECT	LL;

	if (!target->speed) {
		R[X] = R[Y] = R[Z] = 0;
		return;
	}

	LL[X] = p->V[X] - (long)EE(p)->tvnew[X];	/* target rel v */
	LL[Y] = p->V[Y] - (long)EE(p)->tvnew[Y];
	LL[Z] = p->V[Z] - (long)EE(p)->tvnew[Z];
	rv = (int)(lhypot3d (LL)/VONE);			/* my closure v */
	dist = (int)(ldist3d (target->R, p->R)/VONE);	/* target distance */

	if (rv <= dist/8)
		dt = FONE;
	else
		dt = muldiv (FONE/8, dist , rv);
	speed = ihypot3d (EE(p)->tvnew); /* don't trust target->speed */
	if (VMAX/8 <= fmul (dt, speed))
		dt = muldiv (FONE, VMAX/8, speed);
	Vmuldiv (R, EE(p)->tvnew, dt, FONE/8);
}

extern void FAR
SetKillCorrection (OBJECT *p, OBJECT *target, VECT R, VECT A, int *tti)
{
	int	dist, rv, dt, g, speed;
	long	ldist;
	VECT	V;
	LVECT	LL;

	speed = BulletSpeed (p, V);			/* bullet speed */

	Vsub (LL, p->R, target->R);
	ldist = lhypot3d (LL);

	if (ldist)
		rv = speed/VONE - (int)((LL[X]*EE(p)->tvnew[X] +
					 LL[Y]*EE(p)->tvnew[Y] +
					 LL[Z]*EE(p)->tvnew[Z])/(ldist*VONE));
	else
		rv = 0;

	if (ldist > VMAX)
		dist = VMAX/VONE;
	else
		dist = (int)(ldist/VONE);

	if (rv <= dist/32)
		dt = 32*1000;
	else
		dt = muldiv (1000, dist , rv);

	*tti = dt;

	Vsub (A, EE(p)->tvnew, EE(p)->tvold);	/* target accel */

	if (!target->speed && !(st.flags1 & SF_USEG)) {
		R[X] = R[Y] = R[Z] = 0;
		return;
	}

	if (dt < 4000) {
/* projected equivalent speed:
 * 	V+A*t/2
*/
		R[X] = EE(p)->tvnew[X] + muldiv (A[X], dt, st.interval*2);
		R[Y] = EE(p)->tvnew[Y] + muldiv (A[Y], dt, st.interval*2);
		R[Z] = EE(p)->tvnew[Z] + muldiv (A[Z], dt, st.interval*2);
		if (st.flags1 & SF_USEG)
			R[Z] += muldiv (GACC, dt, 2*1000);	/*bullet G!*/
/* projected movement:
 *	V*t
*/
		R[X] = muldiv (R[X], dt, 1000);
		R[Y] = muldiv (R[Y], dt, 1000);
		R[Z] = muldiv (R[Z], dt, 1000);
	} else {
		speed = ihypot3d (EE(p)->tvnew); /* don't trust target->speed */
		if (!speed)
			R[X] = R[Y] = R[Z] = 0;
		else if (VMAX/speed <= dt/1000)
			Vmuldiv (R, EE(p)->tvnew, VMAX, speed);
		else
			Vmuldiv (R, EE(p)->tvnew, dt, 1000);

		if ((st.flags1 & SF_USEG) && dt < 3000) {
			g = muldiv (GACC, dt, 2000);
			g = muldiv (g, dt, 1000);
			R[Z] += g;
		}
	}
}

static void FAR
set_throttle (OBJECT *p, POINTER *ptr, int speed, int interval)
{
	int	throttle;

	if (speed < 0)
		speed = 0;
	throttle = (int)pid_calc (EX->PIDthrottle, speed - (long)p->speed,
					interval);
	if (throttle > 0) {
		if (100 == EX->throttle) {
			if (EX->afterburner)
				ptr->b[0] = throttle/4;
			else if (throttle > 4)
				++ptr->b[17];		/* light AB */
		} else
			ptr->b[0] = throttle;
		if (EE(p)->equip & EQ_AIRBRAKE)
			ptr->b[13] = 1;			/* speed-brakes off */
	} else if (throttle < 0) {
		if (-20 == throttle) {
			if (!(EE(p)->equip & EQ_AIRBRAKE))
				ptr->b[13] = 1;		/* speed-brakes on */
		}
		if (EX->afterburner)
			ptr->b[1] = (-throttle)/4;
		else
			ptr->b[1] = -throttle;
	}
}

static void FAR
go_to (OBJECT *p, int interval, long altitude, ANGLE heading, int speed)
{
	POINTER	*ptr;
	ANGLE	pitch, roll, rdiff, pdiff, hdiff;
	int	pset, hset, rset, tset, ddiff, upright, factor, maxpitch;
	long	ldiff;

	if (!(ptr = p->pointer))
		return;

	if (IS_CLASSIC(p))
		factor = CLFACT;
	else
		factor = F16FACT;

	if (EE(p)->flaps)
		maxpitch = D90/8;
	else {
		maxpitch = p->speed/VONE/2;
		if (maxpitch > LIFTOFF)
			maxpitch = MAXPITCH;
		else
			maxpitch = muldiv (MAXPITCH, maxpitch, LIFTOFF);
	}
	ldiff = altitude - p->R[Z]; 		/* climb rate */
	if (ldiff > MINCLIMB)
		pitch = maxpitch;
	else if (ldiff < -MINCLIMB)
		pitch = -maxpitch;
	else
		pitch = muldiv ((int)ldiff, maxpitch, MINCLIMB);

#define PITCHBAND	D90/4
	pdiff = pitch - p->a[X];		/* pitch rate */
	if (pdiff > PITCHBAND)
		pset = 100;
	else if (pdiff < -PITCHBAND)
		pset = -100;
	else {
#if 0
		pset  = muldiv (pdiff, 100, PITCHBAND);
#else
		pset = (int)pid_calc (EX->PIDpitch, (long)pdiff, interval);
		pset += ptr->l[1];
		if (pset > 100)
			pset = 100;
		else if (pset < -100)
			pset = -100;
#endif
	}

	hdiff = heading - p->a[Z];
	if (hdiff > factor)			/* turn rate */
		hset = 100;
	else if (hdiff < -factor)
		hset = -100;
	else
		hset  = muldiv (hdiff, 100, factor);

	upright = iabs (hdiff) < D90/16 && iabs (pdiff) < D90/16;

	tset = ihypot2d (hset, pset);
	if (0 == tset) {
		roll = 0;
		rdiff = -p->a[Y];
	} else {
		if (pset < 0)
			tset = -tset;
		roll = -ASIN (muldiv (FONE, hset, tset));
		rdiff = roll - p->a[Y];
		if (!upright && iabs (rdiff) > D90) { /* go the short way */
			tset = -tset;
			rdiff += D180;
			roll += D180;
		}
		if (tset < -100) {		/* avoid redout */
			tset = 0;
			rdiff += D180;
			roll += D180;
		}
	}


	if (rdiff > factor)			/* set ailerons */
		rset = -100;
	else if (rdiff < -factor)
		rset = 100;
	else
		rset = -muldiv (rdiff, 100, factor);
	ddiff = rset - ptr->l[0];
	if (ddiff > 25)
		ddiff = 25;
	else if (ddiff < -25)
		ddiff = -25;
	ptr->l[0] += ddiff;


	tset = fmul (pset, COS(p->a[Y])) - fmul (hset, SIN(p->a[Y]));

	if (tset > 100)				/* set elevators */
		tset = 100;
	else if (tset < -100)
		tset = -100;
	ddiff = tset - ptr->l[1];
	if (ddiff > 25)
		ddiff = 25;
	else if (ddiff < -25)
		ddiff = -25;
	ptr->l[1] += ddiff;


	ptr->a[0] = ptr->l[0];
	ptr->a[1] = ptr->l[1];

	set_throttle (p, ptr, speed, interval);
}

static void FAR
set_flaps (OBJECT *p)
{
	int	t, k;
	POINTER	*ptr;

	if (!(ptr = p->pointer))
		return;

	if (EE(p)->flaps) {
		k = LIFTOFF;
		t = k*2 - p->speed/VONE;
		if (t > 0) {
			t = SETFLAPS * t / k;
			if (EE(p)->flaps > t)
				ptr->b[10] = -1;
		} else if (EE(p)->flaps)
			ptr->b[10] = -1;
	}
	if ((EE(p)->equip & EQ_GEAR) && p->R[Z] > 50*VONE)
		ptr->b[15] = 1;
	
}

static void FAR
chase (OBJECT *p, int interval)
{
	POINTER	*ptr;
	ANGLE	rdiff, pdiff, a;
	int	t, speed, pset, rset, kill, factor;
	long	dist;
	OBJECT	*target;
	VECT	R, RR;

	if (!(ptr = p->pointer))
		return;

	if (IS_CLASSIC(p))
		factor = CLFACT;		/* classic */
	else
		factor = F16FACT/2;		/* f16 etc. */

	if ((target = EE(p)->target) && target->id != EE(p)->tid)
		target = 0;

	if ((EE(p)->flags & PF_KILL) && !(EE(p)->flags & PF_CHASE)) {
		if (EE(p)->radar & R_SHOOT)
			++ptr->b[5];
		return;
	}

	if (EE(p)->flags & PF_ONGROUND) {
		EE(p)->equip &= ~EQ_GNDBRAKE;
		if (EE(p)->flaps  < SETFLAPS)
			ptr->b[10] = 1;
		speed = LIFTOFF;
		if (p->speed < speed)
			go_to (p, interval, (IS_CLASSIC(p) ? 0L : 100L) * VONE,
				0, PATROLSPEED*VONE);
		else
			go_to (p, interval, 2000L*VONE, 0, PATROLSPEED*VONE);
		return;
	} else
		set_flaps (p);

	if (EE(p)->flaps) {
		pdiff = D90/8 - p->a[X];
		rdiff = 0;
		a = D90;
		kill = 0;
		speed = PATROLSPEED*VONE;
	} else if (!target) {
		a = p->a[Z];
		if (p->R[Y] >  MAXRANGE && iabs(a) < D90)
			a = D180;
		else if (p->R[Y] < -MAXRANGE && iabs(a) > D90)
			a = 0;
		if (p->R[X] >  MAXRANGE && a < 0)
			a = D90;
		else if (p->R[X] < -MAXRANGE && a > 0)
			a = -D90;
		go_to (p, interval, 2000L*VONE, a, PATROLSPEED*VONE);
		return;
	} else {
		if (EE(p)->flags & PF_KILL)
			SetKillCorrection (p, target, R, RR /*dummy*/,
						&t /*dummy*/);
		else
			SetMeetCorrection (p, target, R);

		R[X] = (int)((target->R[X] + R[X] - p->R[X])/VONE);
		R[Y] = (int)((target->R[Y] + R[Y] - p->R[Y])/VONE);
		R[Z] = (int)((target->R[Z] + R[Z] - p->R[Z])/VONE);
		Mxpose (p->T);
		VMmul (RR, R, p->T);
		Mxpose (p->T);

		pdiff = ATAN (RR[Z], iabs(RR[Y]));	/* pitch error */
		rdiff = ATAN (RR[X], iabs(RR[Z]));	/* roll error */
		t = ihypot2d (RR[Z], RR[X]);
		a = ATAN (t, iabs(RR[Y]));		/* total error */

		dist = ihypot3d (RR) * (long)VONE;
		if ((kill = dist > 0)) {
			if (dist > MAXSPEED*3+MINDIST)
				speed = MAXSPEED;
			else {
				speed = (int)((dist-MINDIST)/3);
				if (speed < MINSPEED)
					speed = MINSPEED;
			}
		} else {
			speed = MINSPEED;
			dist = -dist;
		}
	}

	if (kill && (EE(p)->flags & PF_KILL) && (EE(p)->radar & R_SHOOT))
		++ptr->b[5];

#define CHAOS_BAND	D90/30

	if (a < CHAOS_BAND) {			/* limiting? */
		t = muldiv (a, D90, CHAOS_BAND);
#if 0
		if (t < D90/9)
			t = D90/9;
#endif
		if (rdiff > t)
			rdiff = t;
		else if (rdiff < -t)
			rdiff = -t;
	}
	rset = (int)pid_calc (EX->PIDroll, (long)rdiff, interval);
	if (rset > ptr->l[0]+50)
		ptr->l[0] += 50;
	else if (rset < ptr->l[0]-50)
		ptr->l[0] -= 50;
	else
		ptr->l[0] = rset;

	if ((t = iabs (rdiff)) > D90/5) {
		t = muldiv (D90, D90/5, t);
		if (pdiff > t)
			pdiff = t;
		else if (pdiff < -t)
			pdiff = -t;
	}
	pset = (int)pid_calc (EX->PIDpitch, (long)pdiff, interval);
	pset += ptr->l[1];
	if (pset > 100)
		pset = 100;
	else if (pset < -100)
		pset = -100;
	ptr->l[1] = pset;

	ptr->a[0] = ptr->l[0];
	ptr->a[1] = ptr->l[1];

	set_throttle (p, ptr, speed, interval);
}

extern void FAR
dynamics_auto (OBJECT *p, int interval)
{
	POINTER	*ptr;
	ANGLE	a;
	int	t;

	if (!(ptr = p->pointer))
		return;

	if (EE(p)->flags & (PF_CHASE|PF_KILL)) {
		chase (p, interval);
		return;
	}

	if (!(EE(p)->flags & PF_ONGROUND))
		set_flaps (p);

	if (EE(p)->flags & PF_ONGROUND) {
		EE(p)->equip &= ~EQ_GNDBRAKE;
		if (EE(p)->flaps  < SETFLAPS)
			ptr->b[10] = 1;
		SPEED(p) = PATROLSPEED;
		HEADING(p) = 0;
		t = LIFTOFF;
		if (p->speed >= t)
			ALTITUDE(p) = 2000;
		else
			ALTITUDE(p) = (IS_CLASSIC(p) ? 0 : 100);
	} else if (EE(p)->flaps) {
		;
	} else if ((LIFETIME(p) -= interval) <= 0) {
		LIFETIME(p) = RANDTIME;	/* every few seconds */
#ifndef DEBUG_SPEED
		SPEED(p) = RANDOMSPEED;
#else
		SPEED(p) = (rand()%80 + 20)*20;	/* overflow testing */
#endif
		a = HEADING(p) + (rand()%D90) * 2 - D90;
		if ((p->R[Y] >  MAXRANGE && iabs(a) < D90) ||
		    (p->R[Y] < -MAXRANGE && iabs(a) > D90))
			a = D180 - a;
		if ((p->R[X] >  MAXRANGE && a < 0) ||
		    (p->R[X] < -MAXRANGE && a > 0))
			a = -a;
		HEADING(p) = a;

		t = ALTITUDE(p) + rand()%4000 - 2000;
		if (t > MAXALT)
			t = 2*MAXALT-t;
		else if (t < MINALT)
			t = 2*MINALT-t;
		ALTITUDE(p) = t;
	}

	go_to (p, interval, ALTITUDE(p)*(long)VONE, HEADING(p), SPEED(p)*VONE);
}
