/* --------------------------------- hud.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* paint the Head Up Display.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "fly.h"
#include "hud.h"
#include "plane.h"

static int  FAR show_vv (VIEW *view, OBJECT *p, int orgx, int orgy, int maxx,
	int maxy, int sx, int sy, int c, int clipx, int clipy, int shifty,
	int VVD[2]);
static void FAR show_ww (int orgx, int orgy, int sx, int sy, int color);
static void FAR show_plus (int orgx, int orgy, int tx, int ty, int color);
static void FAR show_border (int orgx, int orgy, int sx, int sy, int shifty,
	int color);
static void FAR alarm_off (void);
static void FAR hud_alarm (OBJECT *p, int orgx, int orgy, int ss, int color,
	int mode);

#undef CP
#undef CW
#undef CS

#define CP	view->viewport
#define CW	view->window
#define CS	view->screen

#define EX	EE(p)

#define	GEARVLIMIT	(150*VONE)

extern void FAR
show_hud (VIEW *view, OBJECT *pov, OBJECT *p, int shift, int orgx, int orgy,
	int maxx, int maxy, int mode)
{
	int	hud, hud1, hud2, big, htype, fa18, hudarea, limit, front, hon;
	int	sx, sy, tx, ty, ttx, tty;
	int	x, y, ss, clipx, clipy, shifty;
	ANGLE	a;
	int	VVD[2];
	struct hud	hudp[1];
/*
 * hud only for planes, in front view
*/
	if (ET_PLANE != CV->e_type || (HDD_HUD != mode && !scenery (mode))) {
		alarm_off ();
		return;
	}

	hud = EE(p)->hud;
	hud1 = EE(p)->hud1;
	hud2 = EE(p)->hud2;
	big = hud & HUD_BIG;
	htype = hud1 & HUD_TYPES;
	fa18 = HUD_FA18 == htype;
	front = !view->viewport->rotz && !view->viewport->rotx;
	x = HDD_FRONT == mode && (hud & HUD_ON) && front;
	hon = x || HDD_HUD == mode;
	limit = (x && (hud1 & HUD_LIMIT)) || HDD_HUD == mode;
/*
 * Set a square HUD
*/
	sx = muldiv (CS->device->lengx, maxx, CS->device->sizex);
	sy = muldiv (CS->device->lengy, maxy, CS->device->sizey);

	if (sx > sy) {		/* wide window */
		sx = muldiv (maxx, sy, sx);
		sy = maxy;
	} else {		/* high window */
		sy = muldiv (maxy, sx, sy);
		sx = maxx;
	}

	if (0 == (tx = (sx+32)/64))
		tx = 1;
	if (0 == (ty = (sy+32)/64))
		ty = 1;

	shifty = EX->misc[17];
	if (HDD_HUD == mode) {
		orgy -= fmul (sy, shifty);
	} else {
		y = sy + fmul (sy, shifty);
		hudarea = fdiv (maxy, y);		/* largest allowed */
		a = DEG2ANG (EE(p)->hudarea);
		y = muldiv (CP->z, SIN (a), COS (a));
		if (y < CP->maxy) {		/* fits in window */
			y = muldiv (y, maxy, CP->maxy);	/* y pixels */
			y = fdiv (y, sy);	/* ratio */
			if (y < hudarea)	/* fits in square */
				hudarea = y;
		}
		sx = fmul (sx, hudarea);
		sy = fmul (sy, hudarea);
	}

	ss = fmul (sx, HUDFONT);
	if (ss < 8)
		ss = 8;

	if (!big && HUD_CLASSIC == htype) {
		x = 3*tx;
		y = 2+tx + num_size (99L, ss);
		if (y > x)
			x = y;
		if (maxx-x < sx) {
			x = maxx-x;
			sy = muldiv (sy, x, sx);
			sx = x;
		}

		y = 2+3*ty+ss;
		if (maxy-y < sy) {
			y = maxy-y;
			sx = muldiv (sx, y, sy);
			sy = y;
		}
	}
	if (0 == (tx = (sx+32)/64))
		tx = 1;
	if (0 == (ty = (sy+32)/64))
		ty = 1;

	shifty = fmul (sy, shifty);

	if (limit) {
		clipx = sx;
		clipy = sy;
	} else {
		clipx = maxx;
		clipy = maxy;
	}

	hudp->maxx = maxx;
	hudp->maxy = maxy;
	hudp->orgx = orgx;
	hudp->orgy = orgy;
	hudp->fg = st.hfg;
	hudp->fgi = st.hfgi;
	hudp->ss = ss;
	hudp->sx = sx;
	hudp->sy = sy;
	hudp->tx = tx;
	hudp->ty = ty;
	hudp->shifty = shifty;
	hudp->clipr =  clipx;
	hudp->clipl = -clipx;
	hudp->clipt =  clipy - shifty;
	hudp->clipb = -clipy - shifty;
	hudp->width = 0;
	hudp->height = 0;

	hud_alarm (p, orgx, orgy, ss, st.hfgi, mode);

	VVD[X] = VVD[Y] = 0;

	if (hon) {
		if (big) {
			ttx = -tx;
			tty = -ty;
		} else {
			ttx = tx;
			tty = ty;
		}
		hudp->ttx = ttx;
		hudp->tty = tty;
/*
 * hud border
*/
		if ((hud1 & HUD_BORDER) && HDD_FRONT == mode)
			show_border (orgx, orgy, sx, sy, shifty, st.gray);
/*
 * heading on upper/lower edge
*/
		show_heading (view, p, sx, sy, maxx, maxy, orgx, orgy,
			ttx, tty, tx, ty, ss, shifty);
/*
 * altitude on right edge
*/
		show_altitude (view, p, sx, sy, maxx, maxy, orgx, orgy,
			ttx, tty, tx, ty, ss);
/*
 * speed on left edge
*/
		show_speed (view, p, sx, sy, maxx, maxy, orgx, orgy,
			ttx, tty, tx, ty, ss);
/*
 * plane symbol
*/
		if (hud & HUD_VV) {
			x = show_vv (view, p, orgx, orgy, maxx, maxy, sx, sy,
					st.hfg, sx, sy, shifty, VVD);
			if (hud2 & HUD_VW) {
				if (x) {
					if ((VVDELAY += st.interval) > VVPERIOD)
						VVDELAY = VVPERIOD;
				} else if (VVDELAY > 0) {
					if ((VVDELAY -= st.interval) < 0)
						VVDELAY = 0;
				}
			} else
				VVDELAY = 0;
			if (VVDELAY > 0 || (EE(p)->equip & EQ_GEAR))
				show_ww (orgx, orgy, sx, sy, st.hfg);
		}
/*
 * pitch ladder is centered on the vv or the waterline mark
*/
		if (hud & HUD_LADDER)
			show_pitch (view, p, sx, sy, maxx, maxy, orgx, orgy,
				ttx, tty, tx, ty, ss, shifty, mode);
/*
 * ILS
*/
		if (hud2 & HUD_ILS)
			show_ils (p, sx, sy, orgx, orgy, ss, shifty);
	}
/*
 * Show radar stuff
*/
	if (HDD_HUD == mode || scenery (mode))
		show_radar (view, p, pov, orgx, orgy, maxx, maxy, tx, ty,
			ss, clipx, clipy, sx, sy, limit ? shifty : 0, VVD,
			mode, hon);
/*
 * ailerons/elevators cursor (helps keypad/mouse mode)
*/
	if ((hud & HUD_CURSOR) && p->pointer) {
		x = orgx + muldiv (-p->pointer->a[0], sx-tx, 100);
		y = orgy + muldiv (-p->pointer->a[1], sy-ty, 100);
		add_line (x + tx, y + ty, T_MOVE);
		add_line (x + tx, y - ty, st.cfg);
		add_line (x - tx, y - ty, st.cfg);
		add_line (x - tx, y + ty, st.cfg);
		add_line (x + tx, y + ty, st.cfg);
	}
/*
 * cross hair
*/
	if ((hud & HUD_PLUS) || (HDD_MAP == mode || HDD_RADAR == mode))
		show_plus (orgx, orgy, tx, ty, st.hfg);
}

static void FAR
show_indicators (int x, int y, int dx)
{
	int	i, xx;

	if (st.flags & SF_BLANKER)
		return;

	xx = stroke_size ("X", dx);
	for (i = 0; i < NIND; ++i) {
		if (st.indicators[i])
			stroke_char (x, y, "DLUSGL***R**********"[i], dx,
				st.indicators[i]);
		x += xx;
	}
}

extern void FAR
show_misc (VIEW *view, OBJECT *p, int maxx, int maxy, int orgx, int orgy,
	int ss)
{
	int	hud, hud1, knots, blink;
	int	xl, xr, y, x0, y0, dm, dd, ch;
	long	t;

	hud = EE(p)->hud;
	hud1 = EE(p)->hud1;
	knots = hud1 & HUD_KNOTS;
	blink = ((int)st.present)&0x0080;

	dd = num_size (9L, ss);

	show_indicators (orgx-maxx, orgy-maxy+ss, dd);

	dm = dd*3/2;
	xr = orgx + maxx - 7*dd;
	xl = xr - 7*dd;
	y = orgy + maxy - 2;
	y0 = ss*3/2;

	if (EE(p)->flags & PF_CHASE) {
		if (EE(p)->target)
			stroke_str (xl, y, "CHASE", ss, st.hfg);
		else
			stroke_str (xl, y, "AUTO", ss, st.hfg);
	}
	if (EE(p)->flags & PF_KILL)
		stroke_str (xr, y, "KILL", ss, st.hfg);
	y -= ss;

	if (EE(p)->equip & EQ_GNDBRAKE)
		stroke_str (xl, y, "BRK", ss, st.hfg);
	if (EE(p)->flags & PF_ONGROUND)
		stroke_str (xr, y, "TAXI", ss, st.hfg);
	else if (EE(p)->equip & EQ_GEAR) {
		if (p->speed < GEARVLIMIT || blink)
			stroke_str (xr, y, "GEAR", ss, st.hfg);
	} else if (EX->misc[12]) {
		int	mm, hh;

		mm = (int)(EX->fuel / (60L*EX->misc[12]));
		hh = mm / 60;
		mm -= 60*hh;
		if (hh > 99)
			hh = mm = 99;
		stroke_str (xr, y, "FT", ss, st.hfg);
		x0 = xr+dm+dd;
		x0 += stroke_num (x0, y, hh, ss, st.hfg);
		x0 += stroke_char (x0, y, ':', ss, st.hfg);
		stroke_frac (x0, y, mm, 2, 0, ss, st.hfg);
	}
	y -= y0;

	stroke_char (xl, y, 'D', ss, st.hfg);
	stroke_num (xl+dm, y, (long)p->damage, ss, st.hfg);
	stroke_char (xr, y, 'F', ss, st.hfg);
	stroke_num (xr+dm, y, EE(p)->fuel/100, ss, st.hfg);
	y -= ss;

	if (EE(p)->spoilers) {
		stroke_str (xl, y, "SPL", ss, st.hfg);
		stroke_num (xl+dm+2*dd, y, EE(p)->spoilers, ss,
				st.hfg);
	}
	if (EE(p)->flaps) {
		stroke_str (xr, y, "FLP", ss, st.hfg);
		stroke_num (xr+dm+2*dd, y, EE(p)->flaps, ss,
				st.hfg);
	}
	y -= ss;

	if (EE(p)->rudder) {
		stroke_str (xl, y, "RDR", ss, st.hfg);
		t = abs(EE(p)->rudder);
		x0 = xl+dm+2*dd;
		x0 += stroke_num (x0, y, t , ss, st.hfg);
		stroke_char (x0, y,
			(EE(p)->rudder <= 0) ? 'R' : 'L', ss,
			st.hfg);
	}
	if (EE(p)->airbrake && blink) {
		stroke_str (xr, y, "AIR", ss, st.hfg);
		stroke_num (xr+dm+2*dd, y, EE(p)->airbrake, ss, st.hfg);
	}
	y -= y0;

	stroke_char (xl, y, 'T', ss, st.hfg);
	t = EE(p)->throttle + EE(p)->afterburner/20;
	stroke_num (xl+dm, y, t, ss, st.hfg);
	stroke_char (xr, y, 'E', ss, st.hfg);
	t = (int)(EE(p)->thrust * 100L / EEP(p)->mil_thrust);
	stroke_num (xr+dm, y, t, ss, st.hfg);
	y -= y0;
	if (EE(p)->hud2 & HUD_NAV) {
/*
 * Show simple x/y from home.
*/
		t = p->R[Y]/(VONE*10);
		if (knots)
			t = t * 54 / 100;
		if (t < 0) {
			ch = 'S';
			t = -t;
		} else
			ch = 'N';
		stroke_char (xl, y, ch, ss, st.hfg);
		stroke_frac (xl+dm, y, t, 0, 2, ss, st.hfg);

		t = p->R[X]/(VONE*10);
		if (knots)
			t = t * 54 / 100;
		if (t < 0) {
			ch = 'W';
			t = -t;
		} else
			ch = 'E';
		stroke_char (xr, y, ch, ss, st.hfg);
		stroke_frac (xr+dm, y, t, 0, 2, ss, st.hfg);
		y -= ss;
/*
 * Show correct latitude/longitude.
*/
		if (p->flags & F_KEEPNAV) {
			t = p->misc[3];
			if (t >= 90*60)
				t =  180*60 - t;
			else if (t < -90*60)
				t = -180*60 - t;
			if (t < 0) {
				ch = 'S';
				t = -t;
			} else
				ch = 'N';
			x0 = xl;
			x0 += stroke_char (x0, y, ch, ss, st.hfg);
			x0 += dd/2;
			x0 += stroke_frac (x0, y, t/60, 2, 0, ss, st.hfg);
			x0 += stroke_char (x0, y, ':',        ss, st.hfg);
			      stroke_frac (x0, y, t%60, 2, 0, ss, st.hfg);

			t = p->misc[1];
			if (t < 0) {
				ch = 'W';
				t = -t;
			} else
				ch = 'E';
			x0 = xr;
			x0 += stroke_char (x0, y, ch, ss, st.hfg);
			x0 += dd/2;
			x0 += stroke_frac (x0, y, t/60, 3, 0, ss, st.hfg);
			x0 += stroke_char (x0, y, ':',        ss, st.hfg);
			      stroke_frac (x0, y, t%60, 2, 0, ss, st.hfg);
			y -= ss;
		}
		stroke_str (xl, y, show_time ("TIME ", st.present), ss, st.hfg);
	}
}

static int FAR
show_vv (VIEW *view, OBJECT *p, int orgx, int orgy, int maxx, int maxy,
	int sx, int sy, int c, int clipx, int clipy, int shifty, int D[2])
{
	int	x, y, tx, ty, rx, ry, type, ret, blink;
	VECT	RR;

	tx = fmul (sx, SVV);
	ty = fmul (sy, SVV*4/5);
	rx = fmul (sx, RVV);
	ry = fmul (sy, RVV);

	type = EE(p)->hud1 & HUD_TYPES;
	if (type == HUD_F16 || type == HUD_F15) {
		tx = rx*2;
		ty = ry*2;
	}

	Vcopy (RR, EE(p)->v);
	screen_coords (view, RR);
	ret = clip_to_screen (D, RR, maxx, maxy, tx, ry, clipx, clipy, shifty);
	if (2 == ret)
		return (1);
	if (D[Y] > clipy-ty-shifty) {
		D[Y] = clipy-ty-shifty;
		ret = 1;
	}
	if (1 == ret) {
		blink = ((int)st.present)&0x080;	/* blink rate: 4/sec */
		if (blink)
			return (1);
	}

	x = orgx + D[X];
	y = orgy - D[Y];

	if (type == HUD_CLASSIC) {
		add_line (x,    y-ry, T_MOVE);
		add_line (x-rx, y,    c);
		add_line (x,    y+ry, c);
		add_line (x+rx, y,    c);
		add_line (x,    y-ry, c);
	} else {
		add_5op (T_ELLIPSE, x, y, rx, ry, c);
		add_line (x,    y-ry, T_MOVE);
	}

	add_line (x,    y-ty, c);
	add_line (x+rx, y,    T_MOVE);
	add_line (x+tx, y,    c);
	add_line (x-rx, y,    T_MOVE);
	add_line (x-tx, y,    c);

	return (ret);
}

static void FAR
show_ww (int orgx, int orgy, int sx, int sy, int color)
{
	int	tx, ty;

	tx = fmul (sx, SVV*6/25);
	ty = fmul (sy, SVV*12/25);

	add_line (orgx - tx*3, orgy,    T_MOVE);
	add_line (orgx - tx*2, orgy,    color);
	add_line (orgx - tx,   orgy+ty, color);
	add_line (orgx,        orgy,    color);
	add_line (orgx + tx,   orgy+ty, color);
	add_line (orgx + tx*2, orgy,    color);
	add_line (orgx + tx*3, orgy,    color);
}

static void FAR
show_plus (int orgx, int orgy, int tx, int ty, int color)
{
	int	x, y, dx, dy;

	x = tx*6;
	y = ty*6;
	dx = tx*2;
	dy = tx*2;

	add_line (orgx - x,  orgy,      T_MOVE);
	add_line (orgx - dx, orgy,      color);
	add_line (orgx + dx, orgy,      T_MOVE);
	add_line (orgx + x,  orgy,      color);

	add_line (orgx,      orgy - y,  T_MOVE);
	add_line (orgx,      orgy - dy, color);
	add_line (orgx,      orgy + dy, T_MOVE);
	add_line (orgx,      orgy + y,  color);
}

static void FAR
show_border (int orgx, int orgy, int sx, int sy, int shifty, int color)
{
	add_line (orgx-sx, orgy-sy+shifty, T_MOVE);
	add_line (orgx+sx, orgy-sy+shifty, color);
	add_line (orgx+sx, orgy+sy+shifty, color);
	add_line (orgx-sx, orgy+sy+shifty, color);
	add_line (orgx-sx, orgy-sy+shifty, color);
}

extern void FAR
show_num (int x, int y, long t, int s, int c, int orgx, int orgy, int maxx,
	int maxy, int shifty)
{
	int	dxs, dxc, dys, dyc, l, h;

	num_extent (t, s, &dxs, &dxc, &dys, &dyc);

	--maxx;			/* fight truncation errors */
	--maxy;

	l = orgx-maxx-x;
	h = orgx+maxx-x;
	if (0 > h || 0 < l)
		return;
	if (dxc > h || dxc < l)
		return;
	if (-dys > h || -dys < l)
		return;
	if (dxc-dys > h || dxc-dys < l)
		return;

	l = orgy+shifty-maxy-y;
	h = orgy+shifty+maxy-y;
	if (0 > h || 0 < l)
		return;
	if (-dxs > h || -dxs < l)
		return;
	if (-dyc > h || -dyc < l)
		return;
	if (-dxs-dyc > h || -dxs-dyc < l)
		return;

	stroke_num (x, y, t, s, c);
}

extern void FAR
add_segment (int x1, int y1, int x2, int y2, int c, int orgx, int orgy,
	int sx, int sy, int shifty)
{
	int	i, z1, z2, xl, xh, yl, yh;
/*
 * not quite midpoint clipping, if both ends our out then we reject the
 * segment which is mostly ok.
*/
	xh = orgx+sx;
	xl = orgx-sx;
	yh = orgy+sy+shifty;
	yl = orgy-sy+shifty;

	z1 = x1>xh || x1<xl || y1>yh || y1<yl;
	z2 = x2>xh || x2<xl || y2>yh || y2<yl;

	if (z1) {
		if (z2)
			return;
		i = x1; x1 = x2; x2 = i;
		i = y1; y1 = y2; y2 = i;
	} else if (!z2) {
		add_line (x1, y1, T_MOVE);
		add_line (x2, y2, c);
		return;
	}

	add_line (x1, y1, T_MOVE);

	i = iabs(x2-x1);
	z1 = iabs(y2-y1);
	if (i < z1)
		i = z1;
	for (; i > 1; i >>= 1) {
		z1 = (x1 + x2)/2;
		z2 = (y1 + y2)/2;
		if (z1>xh || z1<xl || z2>yh || z2<yl) {
			x2 = z1;
			y2 = z2;
		} else {
			x1 = z1;
			y1 = z2;
		}
	}

	add_line (x1, y1, c);
}

extern void FAR
add_dash (int x1, int y1, int x2, int y2, int ndash, int ratio, int c)
{
	int	i, dx, dy, rx, ry;

	if (!ndash)
		return;

	x2 -= x1;
	y2 -= y1;
	ratio /= ndash;
	rx = fmul (x2, ratio);
	ry = fmul (y2, ratio);

	for (i = 0; i < ndash; ++i) {
		dx = x1 + muldiv (x2, i, ndash);
		dy = y1 + muldiv (y2, i, ndash);
		add_line (dx,    dy,    T_MOVE);
		add_line (dx+rx, dy+ry, c);
	}
}

extern void FAR
screen_coords (VIEW *view, VECT RR)
{
	int	s;

	s = CP->z;				/* get minimum */
	if (s > CP->maxx)
		s = CP->maxx;
	if (s > CP->maxy)
		s = CP->maxy;
	RR[X] = muldiv (RR[X], s, CP->maxx);
	RR[Z] = muldiv (RR[Z], s, CP->maxy);
	RR[Y] = muldiv (RR[Y], s, CP->z);
}

extern int FAR
clip_to_screen (int D[2], VECT R, int maxx, int maxy, int dx, int dy,
	int clipx, int clipy, int shifty)
{
	int	off_screen, clip, x, y, ry, rz, t;

	off_screen = 0;			/* some classic clipping */

	if (R[Y] <= 0) {
		ry = -R[Y];
		clip = 0;
	} else
		clip = ry = R[Y];

	if (R[X] >= clip)
		off_screen |= 1;	/* right */
	else if (-R[X] >= clip)
		off_screen |= 2;	/* left */

	rz = R[Z] + muldiv (shifty, ry, maxy);
	if (rz >= clip)
		off_screen |= 4;	/* top */
	else if (-rz >= clip)
		off_screen |= 8;	/* bottom */

	if (off_screen == 5)		/* top right */
		if (R[X] > rz)
			off_screen = 1;
		else
			off_screen = 4;
	else if (off_screen == 9)	/* bottom right */
		if (R[X] > -rz)
			off_screen = 1;
		else
			off_screen = 8;
	else if (off_screen == 6)	/* top left */
		if (-R[X] > rz)
			off_screen = 2;
		else
			off_screen = 4;
	else if (off_screen == 10)	/* bottom left */
		if (-R[X] > -rz)
			off_screen = 2;
		else
			off_screen = 8;
	else
		{}

	switch (off_screen) {
	default:
	case 0:
		if (ry == 0)
			x = y = 0;
		else {
			x = muldiv (maxx, R[X], ry);
			y = muldiv (maxy, rz, ry);
		}
		break;
	case 1:
		x = maxx;
		if (R[X] == 0)
			y = 0;
		else
			y = muldiv (maxy, rz, R[X]);
		break;
	case 2:
		x = -maxx;
		if (R[X] == 0)
			y = 0;
		else
			y = -muldiv (maxy, rz, R[X]);
		break;
	case 4:
		if (rz == 0)
			x = 0;
		else
			x = muldiv (maxx, R[X], rz);
		y = maxy;
		break;
	case 8:
		if (rz == 0)
			x = 0;
		else
			x = -muldiv (maxx, R[X], rz);
		y = -maxy;
		break;
	}
	if (off_screen)
		off_screen = clip ? 1 : 2;
	if (x >= (t = clipx-dx)) {
		y = muldiv (t, y, x);
		x = t;
		if (!off_screen)
			off_screen = 1;
	} else if (x <= (t = dx-clipx)) {
		y = muldiv (t, y, x);
		x = t;
		if (!off_screen)
			off_screen = 1;
	}
	if (y >= (t = clipy-dy)) {
		x = muldiv (t, x, y);
		y = t;
		if (!off_screen)
			off_screen = 1;
	} else if (y <= (t = dy-clipy)) {
		x = muldiv (t, x, y);
		y = t;
		if (!off_screen)
			off_screen = 1;
	}
	D[X] = x;
	D[Y] = y - shifty;
	return (off_screen);
}

extern void FAR
hud_setup (OBJECT *p)
{
	int	font;

	EX->hud  &= HUD_ON|HUD_CURSOR|HUD_DATA|HUD_ROSS;
	EX->hud1 &= HUD_TYPES|HUD_INAME|HUD_IDIST|HUD_MISC;

	EX->hud  |= HUD_DEFAULT|HUD_LADDER;
	EX->hud1 |= HUD_VALARM|HUD_AALARM|HUD_LIMIT|HUD_CORNER|HUD_BORDER;
	EX->hud2 |= HUD_HEADING|HUD_ALTITUDE|HUD_SPEED;

	switch (EX->hud1 & HUD_TYPES) {
	case HUD_F15:
		EX->hud  |= HUD_FINE;
		EX->hud1 |= HUD_KNOTS|HUD_ACCVECT;
		EX->hud2 |= HUD_HIDETGT;
		EX->hudarea = 10;
		EX->misc[17] = FCON (0.30);
		font = 1;
		break;
	case HUD_F16:
		EX->hud1 |= HUD_TOP|HUD_KNOTS|HUD_ACCVECT;
		EX->hudarea = 10;
		EX->misc[17] = FCON (0.25);
		font = 1;
		break;
	case HUD_FA18:
		EX->hud |= HUD_FULLHEADING;
		EX->hud1 |= HUD_KNOTS;
		EX->hud2 |= HUD_VW;
		EX->hudarea = 10;
		EX->misc[17] = FCON (0.4);
		font = 1;
		break;
	default:
		EX->hud  |= HUD_BIG|HUD_XFINE;
		EX->hud1 |= HUD_KNOTS|HUD_TOP;
		EX->hud2 |= HUD_VW;
		EX->hudarea = 13;
		EX->tapelen = 16;		/* scales length */
		EX->misc[17] = FCON (0.3);
		font = 0;
		break;
	}
	if (1 != EP->opt[0])			/* if not classic */
		EX->hud  |= HUD_VV;

	if (!CC || CC == p)
		st.StFont = font;
}

extern char * FAR
get_wname (OBJECT *p)
{
	switch (EE(p)->weapon) {
	case WE_M61:
		return ("M61");
	case WE_MK82:
		return ("MK82");
	default:
		return ("XXX");
	}
}

static void FAR
alarm_off (void)
{
	if (-1 != TnAlarm[0])
		TnAlarm[0] = -1;
	if (-1 != TnWarn[0])
		TnWarn[0] = -1;
}

static void FAR
hud_alarm (OBJECT *p, int orgx, int orgy, int ss, int color, int mode)
{
	int	i, alarm, valarm, balarm, blink, dx;
	long	t;
/*
 * Put high priority alarms last to activate the most urgent aural warning.
*/
	blink = ((int)st.present)&0x0080;
	alarm = 0;
	dx = stroke_size ("A", ss);

	if (ET_PLANE == p->e_type && (p->gpflags & GPF_PILOT)) {
		valarm = ((EE(p)->hud & HUD_ON) && (EE(p)->hud1 & HUD_VALARM))
			 || HDD_HUD == mode;
		balarm = valarm && blink;
		if (st.indicators[4]) {
			alarm = 2;
			if (balarm)
				stroke_str (orgx-dx*6, orgy-ss*5, "GLIMIT",
					ss*2, color);
		}
		if (st.indicators[3]) {
			alarm = 2;
			if (balarm)
				stroke_str (orgx-dx*5, orgy-ss*3, "STALL",
					ss*2, color);
		}
		if (EE(p)->fuel <= 0)
			alarm = 2;
		if (valarm) {
			t = 1000L * EE(p)->fuel / EEP(p)->fuel_capacity;
			t = (t/1000)*1000+1;
			if (t < 10000L && (st.present%t) < 0x080)
				stroke_str (orgx-dx*4, orgy+ss*5, "FUEL",
					ss*2, color);
		}
		CL->next->flags &= ~F_VISIBLE;
		if (!(EE(p)->flags & PF_ONGROUND)) {
			if (1 != EEP(p)->opt[0]) {
				i = muldiv (4000, iabs(p->speed), 300*VONE);
				i = muldiv (i, iabs(p->a[X]), D90);
				if (i < 2000)
					i = 2000;
			} else
				i = 2000;
			t = i * (long)p->V[Z] / 1000;
			if (!(EE(p)->equip & EQ_GEAR) && blink &&
					(p->R[Z] < 0L || p->R[Z]+t/2 < 0L))
				CL->next->flags |= F_VISIBLE;
			if (p->R[Z] < 0L || p->R[Z]+t < 0L) {
				alarm = 1;
				if (balarm)
					stroke_str (orgx-dx*10, orgy+3*ss,
						"PULL UP", ss*3, color);
			}
		}
		if (p->damage <= 0) {
			alarm = 1;
			if (balarm)
				stroke_str (orgx-dx*8, orgy, "EJECT",
					ss*3, color);
		}
		if (!(EE(p)->hud1 & HUD_AALARM))
			alarm = 0;
	}

	switch (alarm) {
	default:
	case 0:
		alarm_off ();
		break;
	case 1:
		if (st.quiet && -1 == TnAlarm[0]) {
			TnAlarm[0] = 0;
			Snd->List (TnAlarm, 0);
		}
		break;
	case 2:
		if (st.quiet && -1 == TnWarn[0]) {
			TnWarn[0] = 0;
			Snd->List (TnWarn, 0);
		}
		break;
	}
}
