/* --------------------------------- init.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* startup code.
*/

#define	VERSION		"1.00[beta]"
#define	VERDATE		__DATE__
#define	VERTIME		__TIME__
#define	COMM_VERSION	0x0005

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "fly.h"

extern struct SysDriver SysDriver;
extern struct TmDriver TmDriver;

static int FAR
get_option (char *e)
{
	char	t, *p;
	int	errs = 0, itemp;
	long	ltemp;

	while (e[0] == ' ')
		++e;
	if (e[0] == '\0')
		return (0);

	for (t = 0, p = e; *p; ++p) {
		if (isspace (*p)) {
			t = *p;
			*p = '\0';
			break;
		}
	}

	LogPrintf (" %s\n", e);
	switch (e[0]) {
	case 'b':
		if (e[1] >= '0' && e[1] <= '9')
			st.big_screen = e[1] - '0';
		else
			st.big_screen = 1;
		break;
	case 'B':
		if (1 != sscanf (e+1, "%lu", &ltemp)) {
			LogPrintf ("use -Bnnn\n");
			++errs;
			break;
		}
		st.maxbuffers = (int)(ltemp/BUFLEN*2+1);
		break;
	case 'd':
		st.vname = xfree (st.vname);
		st.vname = strdup (e+1);
		break;
	case 'D':
		st.dtype = xfree (st.dtype);
		st.dtype = strdup (e+1);
		break;
	case 'f':
		st.fname = xfree (st.fname);
		st.fname = strdup (e+1);
		break;
	case 'F':
		st.fdir = xfree (st.fdir);
		st.fdir = strdup (e+1);
		break;
	case 'H':
		if (1 != sscanf (e+1, "%lx", &st.hud_color)) {
			LogPrintf ("use -Hrrggbb\n");
			++errs;
		}
		break;
	case 'h':
	case '?':
		LogPrintf ("fly8 version %s by Eyal Lebedinsky\n",
			VERSION);
		LogPrintf ("usage: fly8 [options] >>fly.log\n");
		LogPrintf ("       see fly.ini for options\n");
		++errs;
		break;
	case 'I':
		/* already processed */
		break;
	case 'l':
		st.flags ^= SF_LANDSCAPE;
		break;
	case 'L':
		/* already processed */
		break;
	case 'm':
		st.dname = xfree (st.dname);
		st.dname = strdup (e+1);
		break;
	case 'M':
		st.mname = xfree (st.mname);
		st.mname = strdup (e+1);
		break;
	case 'n':
		{
			struct netname	*p;

			if (!NEW (p)) {
				LogPrintf ("no mem for netname\n");
				++errs;
				break;
			}
			p->name = strdup (e+1);
			p->next = st.netnames;
			st.netnames = p;
		}
		break;
	case 'N':
		st.nikname = xfree (st.nikname);
		st.nikname = strdup (e+1);
		if (strlen (st.nikname) >LNAME-1)
			st.nikname[LNAME-1] = '\0';	/* keep it short */
		break;
	case 'p':
		st.pname = xfree (st.pname);
		st.pname = strdup (e+1);
		break;
	case 'P':
		st.ptype = xfree (st.ptype);
		st.ptype = strdup (e+1);
		break;
	case 'q':
		st.quiet = 0;
		break;
	case 'r':
		itemp = 1;
		if (e[itemp] == '0') {
			st.network &= ~NET_ON;
			++itemp;
		} else
			st.network |= NET_ON;
		if (e[itemp] == 'l') {
			++itemp;
			if (e[itemp] == '0') {
				st.network &= ~NET_NOBCAST;
				++itemp;
			} else
				st.network |= NET_NOBCAST;
		}
		break;
	case 's':
		if (1 != sscanf (e+1, "%x", &itemp)) {
			LogPrintf ("use -sPortAddr\n");
			++errs;
		}
		st.misc[7] = itemp;
		break;
	case 't':
		if (1 != sscanf (e+1, "%u", &itemp) ||
			itemp < 0) {
			LogPrintf ("use -tSeconds\n");
			++errs;
		}
		st.ShutdownTime = itemp*1000L;
		break;
	case 'v':
		st.flags ^= SF_VERBOSE;
		break;
	case 'V':
		st.vmdname = xfree (st.vmdname);
		st.vmdname = strdup (e+1);
		break;
	case 'z':
		st.flags ^= SF_BLANKER;
		if (e[1]) {
			if (1 != sscanf (e+1, "%u", &itemp) ||
				itemp < 0) {
				LogPrintf ("use -znn\n");
				++errs;
			}
			st.drones = itemp;
		} else
			st.drones = 0;
		break;
	default:
		LogPrintf ("unknown option %s\n", e);
		++errs;
		break;
	}
	if (t)
		*p = t;
	return (errs);
}

static int
get_argopts (char **a)
{
	int	errs = 0;
	char	*p;

	if (*a)
		LogPrintf ("command line args:\n");

	while ((p = *a++)) {
		if (p[0] != '-') {
			if (p[0] == '?')
				p = "-?";
			else {
				LogPrintf ("unknown argument %s\n", p);
				++errs;
				continue;
			}
		}
		p = strdup (p+1);
		errs += get_option (p);
		p = xfree (p);
	}
	return (errs);
}

static int
get_envopts (void)
{
	char	*e, *p, *q;
	int	errs = 0;

	e = getenv ("FLY8");
	if (e) {
		LogPrintf ("env args:\n");
		e = strdup (e);
		for (p = e; p && *p; p = q) {
			while (*p == ' ')	/* find start */
				++p;
			if (*p == '\0')
				break;
			q = strchr (p, ';');	/* find end */
			if (q)
				*q++ = '\0';	/* end string */
			errs += get_option (p);
		}
		e = xfree (e);
	}
	return (errs);
}

static void FAR
make_fname (char *fname, char *dir, char *name)
{
	strcpy (fname, dir);
	strcat (fname, DIRSEP);
	strcat (fname, name);
}

static FILE * FAR
find_ini (void)
{
	FILE	*ini;
	char	*e, *p, *q, fname[256];

	if (st.iname)
		return (fopen (st.iname, RTMODE));

	st.iname = strdup (INIFILE);

	if ((ini = fopen (st.iname, RTMODE)))
		return (ini);

	if ((e = getenv ("HOME"))) {
		make_fname (fname, e, st.iname);
		if ((ini = fopen (fname, RTMODE)))
			return (ini);
	}

	if ((e = getenv ("PATH"))) {
		e = strdup (e);
		for (p = e; p && *p; p = q) {
			q = strchr (p, ';');	/* find end */
			if (q)
				*q++ = '\0';	/* end string */
			make_fname (fname, p, st.iname);
			if ((ini = fopen (fname, RTMODE)))
				return (ini);
		}
		e = xfree (e);
	}
	return (NULL);
}

static int
get_iniopts (void)
{
	int	errs = 0;
	FILE	*ini;
	char	opt[256];

	if (!(ini = find_ini ()))
		return (0);

	LogPrintf ("%s args:\n", st.iname);
	while (fgets (opt, sizeof (opt), ini)) {
		if (opt[0] == '\n' || opt[0] == '#')
			continue;
		opt[strlen(opt)-1] = '\0';	/* remove '\n' */
		errs += get_option (opt);
	}
	if (ferror (ini)) {
		perror ("error reading init file");
		++errs;
	}

	fclose (ini);
	return (errs);
}

extern void FAR
welcome (void)
{
	MsgPrintf (100, "WELCOME TO FLY8");
	MsgPrintf (100, "By Eyal Lebedinsky");
	MsgPrintf (100, "Version %s %s %s", VERSION, VERDATE, VERTIME);
}

static OBJECT * FAR
create_viewer (int type, int nzoom)
{
	OBJECT	*p;

	if ((p = create_object (O_VIEWER, 1))) {
		p->misc[0] = type;
		save_viewport (p);
		zoom (p->viewport, nzoom);
	} else
		MsgEPrintf (-50, "no viewer %u", type);
	return (p);
}

extern void FAR
initialize (char *argv[])
{
	POINTER	*ptr;
	char	**p;

	st.object_id = 1000;			/* reserve 1-999 */
	st.hud_color = 0x408040L;
	st.paralax = 12;
	st.focus = VMAX;
	st.gap = 64;
	st.quiet = 1;
	st.ComVersion = COMM_VERSION;
	st.gravity = 10*VONE;
	st.big_screen = 1;
	st.flags1 |= SF_USEG;
	st.info = 1;
	st.extview = HDD_RADAR;
	st.maxbuffers = 100;
	st.SkyLines = 50;			/* sky lines */

	for (p = argv; *p; ++p) {
		if ('-' != (*p)[0]) {
		} else if ('I' == (*p)[1]) {
			st.iname = xfree (st.iname);
			st.iname = strdup (&(*p)[2]);
		} else if ('L' == (*p)[1]) {
			st.lname = xfree (st.lname);
			st.lname = strdup (&(*p)[2]);
		}
	}

	if (!st.lname)
		st.lname = strdup (LOGFILE);

	if (log_init ()) {
#ifndef NOSTDERR
		fprintf (stderr, "log init failed\n");
#endif
		die ();
	}

	Sys = &SysDriver;
	if (!Sys || Sys->Init ()) {
		LogPrintf ("system init failed\n");
		die ();
	}

	Tm = &TmDriver;
	if (!Tm || Tm->Init ()) {
		LogPrintf ("timer init failed\n");
		die ();
	}

	if (memory_init ()) {
		LogPrintf ("no memory mgr\n");
		die ();
	}

	LogPrintf ("fly8 start: %s\n", Tm->Ctime ());
	LogPrintf ("Version %s %s %s\n", VERSION, VERDATE, VERTIME);
	LogPrintf ("Written by eyal Lebedinsky: eyal@ise.canberra.edu.au\n");

	if (get_iniopts () || get_envopts () || get_argopts (argv))
		die ();

	srand ((int)Tm->Milli ());		/* don't be boring */

	if (init_funcs ()) {
		LogPrintf ("no funcs\n");
		die ();
	}

	Kbd = keyboard_init (0);
	if (!Kbd || Kbd->Init ()) {
		LogPrintf ("keyboard init failed\n");
		die ();
	}
	Snd = sound_init (0);
	if (!Snd || Snd->Init ()) {
		LogPrintf ("sound init failed\n");
		die ();
	}

	if (!NEW (CS)) {
		LogPrintf ("out of memory [%s(%u)]\n", __FILE__, __LINE__);
		die ();
	}

	if (!NEW (CW)) {
		LogPrintf ("out of memory [%s(%u)]\n", __FILE__, __LINE__);
		die ();
	}

	if (!NEW (CP)) {
		LogPrintf ("out of memory [%s(%u)]\n", __FILE__, __LINE__);
		die ();
	}

	if (!st.vmdname)
		st.vmdname = strdup (VMDFILE);

	if (0 == (Gr = devices_init (st.vname))) {
		LogPrintf ("devices init failed\n");
		die ();
	}

	CS->device = devices_select (st.dname);
	if (CS->device == 0 || Gr->Init (CS->device)) {
		if (st.dname) {
			LogPrintf ("no dev: %s\n", st.dname);
			LogPrintf ("trying default\n");
			CS->device = devices_select (NULL);
		}
		if (CS->device == 0 || Gr->Init (CS->device)) {
			LogPrintf ("no device\n");
			die ();
		}
	}
	CS->FgColor = st.white;
	CS->BgColor = st.black;
	CS->BoColor = st.lgray;
	CS->FontWidth = CS->device->FontWidth;
	CS->FontHeight = CS->device->FontHeight;

	st.tfg = st.white;
	st.tbg = st.black;
	st.wfg = st.green;
	st.cfg = st.red;
	st.hfg = st.hudlow;
	st.hfgi = st.hudhigh;
	Gr->SetPalette (st.hfg, st.hud_color);
	Gr->SetPalette (st.hfgi, hi_color (st.hud_color));
					/* now we are in graphics mode! */
	st.dname = xfree (st.dname);
	st.dname = strdup (CS->device->name);

	set_screen (CS->device->sizex, CS->device->sizey);
	windows_set ();
	set_textwin ();
	set_viewport ();

	Gr->SetTextPos (2,1);

	st.StFontSize = 8;

	if (msg_init ()) {
		LogPrintf ("no messages\n");
		die ();
	}

	if (!st.mname)
		st.mname = strdup (MACFILE);

	if (mac_init ()) {
		LogPrintf ("no macros");
		die ();
	}

	MsgPrintf (-100, "System   %s", Sys->name);
	MsgPrintf (-100, "Timer    %s", Tm->name);
	MsgPrintf (-100, "Graphics %s", Gr->name);
	MsgPrintf (-100, "Gr mode  %s", st.dname);
	MsgPrintf (-100, "Sound    %s", Snd->name);
	MsgPrintf (-100, "Keyboard %s", Kbd->name);

	Gr->SetVisual (0);
	Gr->SetActive (0);

	MsgPrintf (-100, "nBuffers %u", st.maxbuffers);

	if (pointers_init ()) {
		LogPrintf ("no pointers\n");
		die ();
	}

	if (!st.fdir)
		st.fdir = strdup (".");
	else if (!strcmp (st.fdir, DIRSEP))
		st.fdir[0] = '\0';

	if (bodies_init ()) {
		LogPrintf ("no bodies\n");
		die ();
	}

	if (land_init ()) {
		LogPrintf ("no land\n");
		die ();
	}

	ptr = pointer_select (st.pname);
	if (!ptr || ptr->control->Init (ptr)) {
		if (st.pname) {
			if (ptr)
				ptr = pointer_release (ptr);
			MsgEPrintf (-100, "no ptr: %s", st.pname);
			st.pname = xfree (st.pname);
		}
		MsgPrintf (-100, "trying default");
		ptr = pointer_select (NULL);
		if (!ptr  || ptr->control->Init (ptr)) {
			if (ptr)
				ptr = pointer_release (ptr);
			LogPrintf ("no pointer\n");
			die ();
		}
		st.pname = strdup (ptr->name);
	}
	MsgPrintf (-100, "Pointer  %s", ptr->name);

	CO = COT = 0;

	st.options = st.ptype;
	if (!(CC = create_object (O_PLANE, 1))) {
		LogPrintf ("no plane %s\n", st.ptype);
		pointer_release (ptr);
		die ();
	}
	CC->pointer = ptr;
	CC->flags |= F_CC;
	CC->gpflags |= GPF_PILOT;
	CV = CC;

	create_viewer (HDD_FRONT, 0);
	create_viewer (HDD_REAR, 0);
	create_viewer (HDD_MAP, 4);
	create_viewer (HDD_RADAR, 4);
	create_viewer (HDD_TARGET, 3*3);
	create_viewer (HDD_PAN, 3*3);
	create_viewer (HDD_GAZE, 0);
	create_viewer (HDD_CHASE, 0);
	create_viewer (HDD_FOLLOW, 0);

	create_viewer (HDD_RIGHT, 0);
	create_viewer (HDD_LEFT,  0);

	remote_init ();

	welcome ();			/* welcome everybody */
	if (st.quiet)
		Snd->List (TnHello, 0);	/* sing a song... */
	Snd->List (TnEngine, 0);

	if (st.flags & SF_BLANKER) {
		EE(CC)->hud1 &= ~HUD_MISC;
		EE(CC)->radar = R_ON | R_LOCK | (3*R_MODE);
		EE(CC)->flags |= PF_CHASE | PF_KILL;
		EE(CC)->weapon = WE_M61;
		CC->flags |= F_STEALTH;
		clear_text ();
	}

	double_buffer (st.flags1 ^ SF_DBUFFERING);

	st.flags |= SF_INITED;
	if (st.misc[7])
		Gr->Shutters (-1);		/* turn on */

	st.big_bang = Tm->Milli ();
	st.present = 0;
	st.DroneTime = st.present;

	{
		static int	keys[] = {KF_INIT};	/* startup macro */

		mac_interpret (keys, rangeof (keys));
	}
}
