/* --------------------------------- menus.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Menus.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "fly.h"

static MENU MenuYN[] = {
	{'y', "Yes"},
	{'n', "No"},
{'\0', 0}};

/* Select info level
*/

static MENU MenuInfo[] = {
	{'0', "off"},
	{'1', "on"},
	{'2', "none"},
	{'3', "timing"},
	{'4', "stats"},
	{'5', "game"},
{'\0', 0}};

static int FAR
menu_info (void)
{
	int	sel;

	sel = st.info;
	sel = menu_open (MenuInfo, sel);

	switch (sel) {
	case MENU_ABORTED:
	case MENU_FAILED:
		break;
	case 0:
		st.flags1 &= ~SF_INFO;
		break;
	case 1:
		st.flags1 |= SF_INFO;
		break;
	default:
		st.info = sel - 2;
		break;
	}
	if (MENU_FAILED != sel)
		menu_close ();

	return (0);
}

/* Create a random object.
*/

static void FAR
clear_object (int name)
{
	OBJECT	*p;

	for (p = CO; p;) {
		if (p->name == name) {
			if (O_PLANE != name ||
			    (EE(p)->flags & PF_AUTO))
				p->flags |= F_DEL|F_MOD;
		}
		p = p->next;
	}
}

#define DRONEPTR	"p1p05500=random"

extern void FAR
emit_drone (void)
{
	OBJECT	*p;
	POINTER *ptr;

	ptr = pointer_select (DRONEPTR);
	if (!ptr || (ptr->control->Init)(ptr))
		return;
	st.options = st.dtype ? st.dtype : st.ptype;
	if (!(p = create_object (O_PLANE, 1)))
		return;
	p->pointer = ptr;
	p->gpflags |= GPF_PILOT;
	p->R[X] = 10000L*VONE;	/* second runway */
	p->R[Y] = 10000L*VONE;
	EE(p)->flags |= PF_AUTO;
	LIFETIME(p) = 5*1000;	/* let it take off nicely */
	SPEED(p) = 250;
	HEADING(p)  = 0;
	ALTITUDE(p) = 100;
}

static MENU MenuEmit[] = {
	{'t', "target"},	/*  0 */
	{'T', "  del"}, 	/*  1 */
	{'g', "gtarget"},	/*  2 */
	{'G', "  del"}, 	/*  3 */
	{'b', "box"},		/*  4 */
	{'B', "  del"}, 	/*  5 */
	{'-', "dell tgts"},	/*  6 */
	{'d', "drone"}, 	/*  7 */
	{'D', "  del"}, 	/*  8 */
	{'y', "drones"},	/*  9 */
	{'z', "killers"},	/* 10 */
	{'r', "game"},		/* 11 */
{'\0', 0}};

static int FAR
menu_emit (void)
{
	int	sel, i;
	char	msg[80], prompt[80];
	OBJECT	*p;

	sel = 0;
	for (;;) {
		sel = menu_open (MenuEmit, sel);
		if (MENU_FAILED == sel)
			break;

		switch (sel) {
		case 0:
			create_object (O_TARGET, 1);
			break;
		case 1:
			clear_object (O_TARGET);
			st.flags1 &= ~SF_TESTING;
			break;
		case 2:
			create_object (O_GTARGET, 1);
			break;
		case 3:
			clear_object (O_GTARGET);
			st.flags1 &= ~SF_TESTING;
			break;
		case 4:
			create_object (O_BOX, 1);
			break;
		case 5:
			clear_object (O_BOX);
			break;
		case 6:
			clear_object (O_TARGET);
			clear_object (O_GTARGET);
			clear_object (O_BOX);
			st.flags1 &= ~SF_TESTING;
			break;
		case 7:
			emit_drone ();
			break;
		case 8:
			clear_object (O_PLANE);
			st.drones = 0;	 	/* no more drones */
			break;
		case 9:
			for (;;) {
				sprintf (prompt, "planes number[%d]?",
					(int)st.drones);
				getstr (prompt, msg, sizeof (msg) - 1);
				if ('\0' == msg[0])
					break;
				if (1 == sscanf (msg, "%u", &i) && i >= 0) {
					st.drones = i;
					break;
				}
			}
			break;
		case 10:
			for (;;) {
				sprintf (prompt, "killers number[%d]?",
					(int)st.killers);
				getstr (prompt, msg, sizeof (msg) - 1);
				if ('\0' == msg[0])
					break;
				if (1 == sscanf (msg, "%u", &i) &&
				    i >= 0 && i <= st.drones) {
					st.killers = i;
					break;
				}
			}
			break;
		case 11:
			st.info = 3;			/* show time & count */
			st.flags1 |= SF_TESTING|SF_INFO;
			st.ntargets = 10;
			st.nbullets = 0;
			st.test_start = st.present;
			for (p = CO; p;) {	/* kill boxes and targets */
				if (p->name == O_GTARGET || p->name == O_BOX ||
				    p->name == O_TARGET || p->name == O_M61 ||
				    p->name == O_MK82)
					p->flags |= F_DEL|F_MOD;
				p = p->next;
			}
			st.mscore = 0;
			for (i = 0; i < st.ntargets; ++i) {
				if (!create_object (O_TARGET, 1))
					break;
				/* 15 seconds + 25 bullets */
				st.mscore += 15*10 + 25*2;
			}
			(*CC->pointer->control->Key)(CC->pointer, KF_ORIGIN);
			(*CC->pointer->control->Key)(CC->pointer, KF_POWER_0);
			(*CC->pointer->control->Key)(CC->pointer, KF_LEVEL);
			break;
		default:
			break;
		}
		menu_close ();
		if (MENU_ABORTED == sel)
			break;
	}

	return (0);
}

/* options menu
*/

static MENU MenuOpts[] = {
	{'v', "Version"},	/*  0 */
	{'s', "Smoke"}, 	/*  1 */
	{'d', "Debug"}, 	/*  2 */
	{'f', "Font"},		/*  3 */
	{'m', "Modes"}, 	/*  4 */
	{'k', "Sky"},		/*  5 */
	{'g', "Gravity"},	/*  6 */
	{'i', "Instuments"},	/*  7 */
	{'b', "Play Blues"},	/*  8 */
	{'V', "Verbose"},	/*  9 */
	{'n', "Net Stats"},	/* 10 */
{'\0', 0}};

static int FAR
menu_opts (void)
{
	int	sel;

	sel = menu_open (MenuOpts, 0);

	switch (sel) {
	default:
	case MENU_ABORTED:
	case MENU_FAILED:
		break;
	case 0:
		welcome ();
		break;
	case 1:
		st.flags1 ^= SF_SMOKE;
		break;
	case 2:
		st.flags ^= SF_DEBUG;
		break;
	case 3:
		st.flags ^= SF_FONT;
		break;
	case 4:
		if (!(st.flags & SF_MODES))
			st.flags &= ~SF_LISTS;
		st.flags ^= SF_MODES;
		break;
	case 5:
		st.flags ^= SF_SKY;
		if (st.flags & SF_SKY)
			sky_init ();
		else
			sky_term ();
		break;
	case 6:
		st.flags1 ^= SF_USEG;
		break;
	case 7:
		if (ET_PLANE != CV->e_type)
			break;
		EE(CV)->flags ^= PF_INSTRUMENTS;
		break;
	case 8:
		if (st.quiet)
			Snd->List (TnTune4, 0);
		break;
	case 9:
		st.flags ^= SF_VERBOSE;
		break;
	case 10:
		if (!(st.flags & SF_NET))
			st.flags &= ~SF_LISTS;
		st.flags ^= SF_NET;
		break;
	}
	if (MENU_FAILED != sel)
		menu_close ();

	return (0);
}

/* net menu
*/

static PLAYER FAR no_player[1] = {{0}};
static PLAYER FAR all_players[1] = {{0}};

/* returns: selected player, 0 (abort), no_player or all_players.
*/
static PLAYER * FAR
choose_player (int mode)
{
	int	i, j;
	PLAYER	*pl;
	char	msg[30];

	for (i = 0, pl = 0; (pl = player_next (pl));) {
		if (pl->flags & mode)
			MsgPrintf (100, "%u %s", ++i, pl->name);
	}
	if (!i)
		return (no_player);
	do {
		if (i > 1) {
			getstr ("choose player", msg, sizeof (msg) - 1);
			pl = 0;
			if ('*' == msg[0])
				return (all_players);
			if ('\0' == msg[0])
				return (0);
			if (1 == sscanf (msg, "%u", &j))
				break;
		} else
			j = i;
		for (i = 0, pl = 0; (pl = player_next (pl));) {
			if (pl->flags & mode)
				if (++i == j)
					break;
		}
	} while (!pl);

	if (j == 0)
		return (0);
	else
		return (pl);
}

static MENU MenuNet[] = {
	{'p', "ping"},		/*  0 */
	{'y', "play"},		/*  1 */
	{'q', "quit"},		/*  2 */
	{'m', "message"},	/*  3 */
	{'a', "accept"},	/*  4 */
	{'d', "decline"},	/*  5 */
	{'A', "auto accept"},	/*  6 */
	{'D', "auto decline"},	/*  7 */
	{'q', "manual reply"},	/*  8 */
{'\0', 0}};

static int FAR
menu_net (void)
{
	int	sel, i;
	char	msg[80];
	PLAYER	*pl;

	if (!(st.network & NET_ON)) {
		MsgPrintf (50, "no net");
		return (0);
	}

	sel = menu_open (MenuNet, 0);

	switch (sel) {
	default:
	case MENU_ABORTED:
	case MENU_FAILED:
		break;
	case 0: 		/* ping */
		remote_ping ();
		break;
	case 1: 		/* play */
		pl = choose_player (~RMT_PLAYING);
		if (pl == no_player)
			MsgPrintf (50, "no players");
		else if (pl == all_players) {
			for (pl = 0; (pl = player_next (pl));) {
				if (!(pl->flags & RMT_PLAYING)) {
					MsgPrintf (50, "Querying %s",
								pl->name);
					remote_play (pl);
				}
			}
		} else if (pl) {
			MsgPrintf (50, "Querying %s", pl->name);
			remote_play (pl);
		}
		break;
	case 2: 		/* quit */
		pl = choose_player (RMT_PLAYING);
		if (pl == no_player)
			MsgPrintf (50, "no players");
		else if (pl == all_players) {
			MsgPrintf (50, "Quitting all");
			remote_noplay (0, 1);
			players_remove ();
		} else if (pl) {
			MsgPrintf (50, "Quitting %s", pl->name);
			remote_noplay (pl, 0);
			player_remove (pl);
		}
		break;
	case 3: 		/* message */
		pl = choose_player (~0);
		if (pl == no_player)
			MsgPrintf (50, "no players");
		else if (pl) {
			getstr ("message text", msg, sizeof (msg) - 1);
			if (pl == all_players) {
				remote_time (0, 1);
				remote_msg (msg, 0, 1);
			} else {
				remote_time (pl, 0);
				remote_msg (msg, pl, 0);
			}
		}
		break;
	case 4:			/* accept */
	case 5:			/* decline */
		i = (4 == sel);
		pl = choose_player (RMT_PENDBOSS);
		if (pl == no_player)
			MsgPrintf (50, "no players");
		if (pl == all_players) {
			for (pl = 0; (pl = player_next (pl));) {
				if (pl->flags & RMT_PENDBOSS)
					remote_reply (pl, i);
			}
		} else if (pl)
			remote_reply (pl, i);
		break;
	case 6:			/* auto accept */
		st.network &= ~NET_AUTOREPLY;
		st.network |= NET_AUTOACCEPT;
		break;
	case 7:			/* auto decline */
		st.network &= ~NET_AUTOREPLY;
		st.network |= NET_AUTODECLINE;
		break;
	case 8:			/* manual reply */
		st.network &= ~NET_AUTOREPLY;
		break;
	}
	if (MENU_FAILED != sel)
		menu_close ();

	return (0);
}

extern void FAR
set_small_frame (void)
{
	CW->orgx -= CW->maxx;		/* left frame */
	CW->maxx = muldiv (CW->maxx, st.gap-1,2*st.gap);
	CW->orgx += CW->maxx;
	CP->maxx = muldiv (CP->maxx, st.gap-1,2*st.gap);
	CP->z	 = muldiv (CP->z, st.gap-1, 2*st.gap);
	adjust_viewports ();
}

static void FAR
set_large_frame (void)
{
	CW->orgx -= CW->maxx;
	if (CW->maxx > FONE/2)
		CW->maxx = FONE;
	else
		CW->maxx = muldiv (CW->maxx, 2*st.gap, st.gap-1);
	CW->orgx += CW->maxx;
	if (CP->maxx > FONE/2)
		CP->maxx = FONE;
	else
		CP->maxx = muldiv (CP->maxx, 2*st.gap, st.gap-1);
	if (CP->z > FONE/2)
		CP->z = FONE;
	else
		CP->z = muldiv (CP->z, 2*st.gap, st.gap-1);
	adjust_viewports ();
}

/* stereo menu
*/

static MENU MenuStr[] = {
	{'m', "Mono"},		/*  0 */
	{'s', "S'Scopic"},	/*  1 */
	{'r', "RedBlue"},	/*  2 */
	{'a', "Alternate"},	/*  3 */
{'\0', 0}};

static int FAR
menu_stereo (void)
{
	int	sel, ret;
	Ushort	flags;

	for (ret = 0; !ret;) {
		sel = menu_open (MenuStr, st.stereo);
		if (MENU_FAILED != sel)
			menu_close ();

		switch (sel) {
		case MENU_ABORTED:
		case MENU_FAILED:
			ret = 1;
			break;
		case 3:
			if (!st.misc[7]) {
				MsgPrintf (50, "No shutters");
				break;
			}
		default:
			flags = st.flags;
			st.flags |= SF_SIMULATING;
			show_fixed (1);
			if (3 == sel)
				Gr->Shutters (-1);	/* turn shutters on */
			else
				Gr->Shutters (-2);	/* turn shutters off */
			if (1 == sel && 1 != st.stereo)
				set_small_frame ();
			else if (1 != sel && 1 == st.stereo)
				set_large_frame ();
			st.stereo = sel;
			show_fixed (0);
			st.flags = flags;
			ret = 1;
			break;
		}
	}

	return (0);
}

/* screen menu
*/

static MENU MenuScr[] = {
	{'c', "Colors"},	/*  0 */
	{'s', "Stereo"},	/*  1 */
	{'r', "Reverse"},	/*  2 */
	{'p', "Paralax"},	/*  3 */
	{'d', "Dbl Buff"},	/*  4 */
	{'b', "Blanker"},	/*  5 */
	{'l', "Load font"},	/*  6 */
	{'f', "Font show"},	/*  7 */
{'\0', 0}};

static int FAR
menu_screen (void)
{
	int	sel, ch, i;
	Ushort	flags;
	char	msg[80];
	HMSG	*m;
	FILE	*ffile;

	sel = 0;
	sel = menu_open (MenuScr, sel);

	switch (sel) {
	default:
	case MENU_ABORTED:
	case MENU_FAILED:
		break;
	case 0:
		if (st.stereo == 2) {
			MsgWPrintf (50, "Cannot in stereo");
			break;
		}
		menu_colors ();
		break;
	case 1:
		menu_stereo ();
		break;
	case 2:
		st.flags1 ^= SF_STEREOREV;
		break;
	case 3:
		for (;;) {
			m = MsgPrintf (0, "paralx(%d) ?[+-]",
				(int)st.paralax);
			ch = mgetch ();
			msg_del (m);
			if (ch == '+')
				++st.paralax;
			else if (ch == '-')
				--st.paralax;
			else if (ch == KF_ESC)
				break;
		}
		break;
	case 4:
		double_buffer (st.flags1 ^ SF_DBUFFERING);
		break;
	case 5:
		flags = st.flags;
		st.flags |= SF_SIMULATING;
		if (st.flags & SF_BLANKER) {
			st.flags ^= SF_BLANKER;
			show_fixed (0);
		} else {
			show_fixed (1);
		}
		st.flags = flags ^ SF_BLANKER;
		break;
	case 6:
		getstr ("Font File?", msg, sizeof (msg) - 1);
		if ('\0' == msg[0])
			break;
		if (!(ffile = fopen (msg, "rb")))
			MsgEPrintf (-50, "cannot open: %s", msg);
		else {
			fclose (ffile);
			st.fname = xfree (st.fname);
			st.fname = strdup (msg);
			Gr->FontSet (CS->device, st.fname);
		}
		break;
	case 7:
		clear_text ();
		Gr->SetTextPos (5, 1);
		for (i = 0; i < 256; ++i) {
			Gr->TextChar (i);
			if (i%16 == 15)
				Gr->TextPut ('\n');
		}
		break;
	}
	if (MENU_FAILED != sel)
		menu_close ();

	return (0);
}

/* Top menu
*/

static MENU MenuTop[] = {
	{'x', "Exit"},		/*  0 */
	{'h', "Help"},		/*  1 */
	{'p', "Pointer"},	/*  2 */
	{'s', "Screen"},	/*  3 */
	{'w', "Windows"},	/*  4 */
	{'i', "Info"},		/*  5 */
	{'e', "Emit"},		/*  6 */
	{'u', "Hud"},		/*  7 */
	{'n', "Net"},		/*  8 */
	{'o', "Options"},	/*  9 */
{'\0', 0}};

extern int FAR
menu_top (void)
{
	int	sel, ret, i;

	sel = menu_open (MenuTop, 0);

	ret = 0;

	switch (sel) {
	default:
	case MENU_ABORTED:
	case MENU_FAILED:
		break;
	case 0:
		if (MENU_FAILED != (i = menu_open (MenuYN, 1))) {
			if (0 == i)	/* "Yes" */
				ret = 1;
			menu_close ();
		}
		break;
	case 1:
		if (!(st.flags & SF_HELP))
			st.flags &= ~SF_LISTS;
		st.flags ^= SF_HELP;
		break;
	case 2:
		menu_ptrs ();
		break;
	case 3:
		menu_screen ();
		break;
	case 4:
		menu_windows ();
		break;
	case 5:
		menu_info ();
		break;
	case 6:
		menu_emit ();
		break;
	case 7:
		if (ET_PLANE != CV->e_type)
			break;
		menu_hud ();
		break;
	case 8:
		menu_net ();
		break;
	case 9:
		menu_opts ();
		break;
	}
	if (MENU_FAILED != sel)
		menu_close ();

	return (ret);
}
