/* --------------------------------- slip.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for packet level exchanges (low level). It uses a packet driver
 * as the communications medium. It expects SLIP type packets.
 * Options:
 *  0 interrupt number (usualy 0x65)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#include "fly.h"
#include "pktdrvr.h"

#define	PKSSIZE	1024		/* packet-driver's stack size */

#define	LDATA		(PACKHEADLEN+ONEPACKLEN)
#define PHLEN		(2*LADDRESS+2+2)
#define PHEAD		(pack->data+PACKHEADLEN-PHLEN)

typedef struct port PORT;
struct port {
	int	flags;
#define POF_ON		0x0001
	void (INTERRUPT FAR *pkint) (void);
	int	intno;
	int	netport;
	Uchar	address[LADDRESS];
	int	handle;
	PACKET	*pack;
	int	*stack;
	int	version;
	int	class;
	int	type;
	int	number;
	char	*name;
	int	basic;
};

static PORT	ports[] = {
	{0, pkint0},
	{0, pkint1},
	{0, pkint2},
	{0, pkint3}
};
#define	NDEV	(sizeof(ports)/sizeof(ports[0]))

static int	nports = 0;		/* number of active ports */

static void FAR
receiver (int dev, Ushort di, Ushort si, Ushort bp, Ushort dx, Ushort cx,
	Ushort bx, Ushort ax, Ushort ds, Ushort es)
{
	PORT	*port;
	PACKET	*pack;
	char	*buff;

	switch (ax) {
	case 0:				/* Space allocate call */
		es = di = 0;
		if (dev < 0 || dev >= NDEV || cx < 3 || cx > PAKPACKLEN)
			goto badret;
		port = &ports[dev];
		if (!(port->flags & POF_ON))
			goto badret;
		if (port->pack) {
			/* stats... */
			packet_del (port->pack);
			port->pack = 0;
		}
		if (!(pack = packet_new ((cx > ONEPACKLEN) ? PAKPACKLEN : 0)))
			goto badret;
		port->pack = pack;
		pack->length = cx;
		buff = PHEAD;
		es = FP_SEG (buff);
		di = FP_OFF (buff);
		break;
	case 1:				/* Packet complete call */
		if (dev < 0 || dev >= NDEV)
			goto badret;
		port = &ports[dev];
		if (!(port->flags & POF_ON))
			goto badret;
		if (!(pack = port->pack))
			goto badret;
		pack->netport = port->netport;
		pack->address = port->address;		/* from */

		if (packet_deliver (pack))
			packet_del (pack);
		port->pack = 0;
		break;
	default:
badret:
		++st.stats[5];
		break;
	}
}

static int FAR
find_driver (void)
{
	int	i;

	for (i = 0x0060; i < 0x0080; ++i)
		if (test_for_pd (i))
			return (i);
	return (-1);
}

static int FAR
get_options (PORT *port, char *options)
{
	port->intno = (int) get_inarg (options, 0);

	return (0);
}

static int FAR
com_init (NETPORT *np, char *options)
{
	int	portno;
	PORT	*port;
	Uchar	unit[2];

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV) {
		MsgEPrintf (-100, "%s.%c: bad port",
			np->NetDriver->name, np->unit);
		return (1);
	}
	port = &ports[portno];
	if (port->flags & POF_ON) {
		MsgEPrintf (-100, "%s: already on", port->address);
		return (1);
	}

	memset  (port->address, 0, LADDRESS);
	strncpy (port->address, np->NetDriver->name, LADDRESS);
	strncat (port->address, ".", LADDRESS);
	unit[0] = (Uchar)np->unit;
	unit[1] = '\0';
	strncat (port->address, unit, LADDRESS);

	if (get_options (port, options))
		return (1);

	if (-1 == port->intno)
		port->intno = find_driver ();
	else if (!test_for_pd (port->intno))
		port->intno = -1;
	if (-1 == port->intno) {
		MsgEPrintf (-100, "%s: no driver", port->address);
		return (1);
	}
	MsgPrintf (-100, "Intno 0x%x", port->intno);

	if (!(port->stack = (int *)xcalloc (PKSSIZE, sizeof (int)))) {
		MsgEPrintf (-100, "%s: no mem", port->address);
		return (1);
	}
	pkset (portno, receiver, &port->stack[PKSSIZE]);

	port->handle = access_type (port->intno, CL_SERIAL_LINE, ANYTYPE, 0,
		(char FAR *)0, 0, port->pkint);
	if (-1 == port->handle) {
		MsgEPrintf (-100, "%s: no handle", port->address);
		port->stack = xfree (port->stack);
		return (1);
	}
	if (driver_info (port->intno, port->handle, &port->version,
			&port->class, &port->type, &port->number, &port->name,
			&port->basic)) {
		port->basic = 1;	/* what else ? */
	}
	MsgPrintf (-100, "Basic 0x%x", port->basic);

	port->flags |= POF_ON;
	port->netport = np->netport;
	port->pack = 0;
	++nports;

	return (0);
}

static void FAR
com_term (NETPORT *np)
{
	int	portno;
	PORT	*port;

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV)
		return;
	port = &ports[portno];
	if (!(port->flags & POF_ON))
		return;
	release_type (port->intno, port->handle);
	port->flags = 0;
	port->stack = xfree (port->stack);
}

static int FAR
com_send (NETPORT *np, PACKET *pack)
{
	int	portno;
	PORT	*port;
	PACKET	*p;

	if (0 == pack)
		return (0);

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV)
		return (1);
	port = &ports[portno];
	if (!(port->flags & POF_ON))
		return (1);
	if (port->basic >= 5) {
		if (!(p = packet_new (0)))
			return (1);
		memcpy (p, pack, sizeof (*p));
		p->next = np->outgoing;
		np->outgoing = p;
		return (as_send_pkt (port->intno, PHEAD, p->length, pksends));
	} else
		return (send_pkt (port->intno, PHEAD, pack->length));
}

static void FAR
com_receive (NETPORT *np)
{
	return;
}

extern struct NetDriver NetSlip = {
	"SLIP",
	0,
	com_init,
	com_term,
	com_send,
	com_receive
};

