/* --------------------------------- netport.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Manage network ports (devices, cards, whatever you want to call these)
 * Only one port is available on most machines.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"

#define PDATA(p)	((p)->data+PACKHEADLEN)

extern struct NetDriver *NetDrivers[];

static NETPORT	*netports = 0;
static int	nnetports = 0;

static int FAR
packet_try (PACKET *pack, NETPORT *np)
{
	if ((np->NetDriver)->Send (np, pack)) {
		if (pack)
			++st.stats[8];
		return (1);
	}
	if (pack) {
		st.stats[0] += pack->length;
		++st.stats[1];
	}
	return (0);
}

static int FAR
netport_init (int n, char *ndname)
{
	NETPORT			*np;
	struct NetDriver	**nd;
	char			*options;
	int			nlen;
	int			unit;

	if (n >= nnetports) {
		MsgEPrintf (-100, "netport: bad port %d", n);
		return (1);
	}

	if (!(nlen = strlen (ndname))) {
		MsgEPrintf (-100, "netport: no name");
		return (1);
	}

	unit = 0;
	options = strchr (ndname, '.');
	if (options) {
		nlen = options - ndname;
		unit = options[1];
		if (!nlen || !unit) {
			MsgEPrintf (-100, "netport: bad name %s", ndname);
			return (1);
		}
		options += 2;
	} else
		options = ndname;

	options = strchr (options, ':');
	if (options) {
		if (!nlen) {
			nlen = options - ndname;
			if (!nlen) {
				MsgEPrintf (-100, "netport: bad options %s",
					ndname);
				return (1);
			}
		}
	}

	for (nd = NetDrivers; *nd; ++nd) {
		if (!strnicmp ((*nd)->name, ndname, nlen))
			break;
	}
	if (!(*nd)) {
		MsgEPrintf (-100, "netport: unknown driver %s", ndname);
		return (1);
	}

	np = &netports[n];
	np->flags = 0;
	np->netport = n;
	np->unit = unit;
	np->NetDriver = *nd;
	np->incoming[HEAD] = np->incoming[TAIL] = 0;

	MsgPrintf (-100, "Netport  %s.%c", (*nd)->name, np->unit);

	if ((*nd)->Init (np, options))
		return (1);
	np->flags |= NP_ON;
	if (get_arg (options, "pack")) {
		np->flags |= NP_PACKING;
		MsgPrintf (-100, "%s.%c packing", (*nd)->name, np->unit);
	}

	return (0);
}

extern int FAR
netports_init (void)
{
	int		n, i;
	struct netname	*nn;

	for (n = 0, nn = st.netnames; nn; ++n, nn = nn->next)
		;

	if (!(netports = (NETPORT *) xcalloc (n, sizeof (NETPORT)))) {
		MsgEPrintf (-100, "netport: no mem");
		return (1);
	}
	nnetports = n;

	for (i = 0, n = 0, nn = st.netnames; nn; ++n, nn = nn->next)
		if (!netport_init (n, nn->name))
			++i;
	if (!i) {
		MsgEPrintf (-100, "netport: no active ports");
		return (1);
	}
	return (0);
}

static void FAR
netport_term (int n)
{
	NETPORT		*np;
	PACKET		*pack;

	np = &netports[n];
	if (!(np->flags & NP_ON))
		return;
	if (np->outpak) {
		if (np->outpak->length > 1)
			packet_try (np->outpak, np);	/* flush packed */
		packet_del (np->outpak);
		np->outpak = 0;
	}
	(np->NetDriver)->Term (np);
	np->flags = 0;
	np->NetDriver = 0;

	for (pack = np->incoming[HEAD]; pack;)
		pack = packet_del (pack);
	np->incoming[HEAD] = np->incoming[TAIL] = 0;

	for (pack = np->outgoing; pack;)
		pack = packet_del (pack);
	np->outgoing = 0;
}

extern void FAR
netports_term (void)
{
	int	n;

	for (n = 0; n < nnetports; ++n)
		netport_term (n);

	netports = xfree (netports);
	nnetports = 0;
}

static void FAR
packet_accept (PACKET *pack)
{
	PLAYER	*pl;
	int	lcrc;

	++st.stats[53];		/* count rcvd packet */
	lcrc = (PDATA (pack)[pack->length-2] << 8)
		      + PDATA (pack)[pack->length-1];
	if (lcrc != crc (pack)) {
		packet_del (pack);
		pack = 0;
		++st.stats[6];
	} else if ((pl = player_active (pack))) {
		pack->next = 0;
		if (pl->tail)
			pl->tail->next = pack;
		else
			pl->incoming = pack;
		pl->tail = pack;
	} else {
		packet_del (pack);	/* oops, drop the packet */
		pack = 0;
		++st.stats[7];
	}
}

static void FAR
packet_packed (PACKET *packed)
{
	Uchar	*p;
	PACKET	*pack;
	int	l;

	p = PDATA(packed) + 1;
	for (l = packed->length - 1; l > 0;) {
		if (!(pack = packet_new (0)))
			break;				/* loose it */
		pack->netport = packed->netport;	/* copy player's id */
		pack->address = packed->address;	/* note temp ptr */
		pack->arrived = packed->arrived;
		pack->length  = (p[0] << 8) + p[1];
		p += 2;
		l -= pack->length + 2;
		if (l < 0) {
			packet_del (pack);
			++st.stats[52];			/* bad packing */
			break;
		}
		memcpy (PDATA(pack), p, pack->length);
		packet_accept (pack);
		pack->address = 0;
		p += pack->length;
	}
	packet_del (packed);
}

static void FAR
netport_receive (NETPORT *port)
{
	PACKET	*pack, *prev;
	short	iflags;

	if (!port || !(port->flags & NP_ON))
		return;

	(port->NetDriver)->Receive (port);

	for (;;) {
		iflags = Sys->Disable ();
		if ((pack = port->incoming[HEAD]) &&
		    !(port->incoming[HEAD] = pack->next))
			port->incoming[TAIL] = 0;
		Sys->Enable (iflags);
		if (!pack)
			break;
		if (RC_PACKED == *PDATA(pack))
			packet_packed (pack);
		else
			packet_accept (pack);
	}

	for (prev = 0, pack = port->outgoing; pack;) {
		if (RC_SENDOK == PDATA (pack)[0])
			;
		else if (RC_SENDFAIL == PDATA (pack)[0])
			++st.stats[9];
		else {			/* send pending */
			prev = pack;
			pack = pack->next;
			continue;
		}
		if (prev)
			prev->next = pack->next;
		pack = packet_del (pack);
	}
	if (!prev)
		port->outgoing = 0;
}

extern void FAR
netports_receive (void)
{
	NETPORT		*port;
	int		i;

	for (i = 0; i < nnetports; ++i) {
		port = &netports[i];
		if (port->flags & NP_ON)
			netport_receive (port);
	}
}

extern void FAR
PlName (PLAYER *pl)
{
	memset (pl->name, 0, LNAME);
#if 1
	sprintf (pl->name, "%02x%02x%02x",
		pl->address[3], pl->address[4], pl->address[5]);
#else
	NETPORT	*port;
	char	unit[2];

	port = &netports[pl->netport];
	if (!(port->flags & NP_ON))
		return;
	unit[0] = (char)port->unit;
	unit[1] = '\0';
	strncpy (pl->name, port->NetDriver->name, LNAME);
	strncat (pl->name, ".", LNAME);
	strncat (pl->name, unit, LNAME);
#endif
}

extern void FAR
netport_count (PLAYER *pl, int delta)
{
	NETPORT	*port;

	port = &netports[pl->netport];
	if (!(port->flags & NP_ON))
		return;
	port->nplayers += delta;
}

extern int FAR
packet_deliver (PACKET *pack)
{
	NETPORT	*port;
	short	iflags;

	port = &netports[pack->netport];
	if (!(port->flags & NP_ON))
		return (1);

	pack->flags = 0;
	pack->next = 0;
	pack->arrived = st.lasttime;

	iflags = Sys->Disable ();
	if (port->incoming[TAIL])
		port->incoming[TAIL]->next = pack;
	else
		port->incoming[HEAD] = pack;
	port->incoming[TAIL] = pack;
	Sys->Enable (iflags);
	st.stats[2] += pack->length;
	++st.stats[3];
	return (0);
}

static int FAR
packet_dispatch (PACKET *pack, int port, int mode)
{
	NETPORT	*np;
	PACKET	*outpak;
	int	ret;

	np = &netports[port];
	if (!(np->flags & NP_ON))
		return (1);
	outpak = np->outpak;
	if (!pack) {
		ret = 0;
		if (outpak && outpak->length > 1) {
			ret |= packet_try (outpak, np);
			outpak->length = 1;	/* reuse outpak */
		}
		ret |= packet_try (pack, np);		/* flush port */
		return (ret);
	}
	++st.stats[54];
	if (!mode || !(np->flags & NP_PACKING))
		return (packet_try (pack, np));
	if (!outpak) {
		outpak = np->outpak = packet_new (PAKPACKLEN);
		if (!outpak)			/* sorry, no can pak */
			return (packet_try (pack, np));
		outpak->length = 1;
		*PDATA(outpak) = RC_PACKED;
	}
	if ((Uint)(pack->length+2) > outpak->size-outpak->length) {
		if (packet_try (outpak, np))
			return (1);
		outpak->length = 1;		/* reuse outpak */
	}
	PDATA(outpak)[outpak->length++] = (Uchar)(0x0ff&(pack->length >> 8));
	PDATA(outpak)[outpak->length++] = (Uchar)(0x0ff&(pack->length));
	memcpy (&PDATA(outpak)[outpak->length], PDATA(pack), pack->length);
	outpak->length += pack->length;
	return (0);
}

extern int FAR
packet_send (PACKET *pack, int mode)
/*
 * mode:
 * 0 send to addressed player
 * 1 broadcast to all netports
 * 2 broadcast to all active netports
*/
{
	int	port, ret;

	ret = 0;
	if (!pack) {
		for (port = 0; port < nnetports; ++port)
			ret |= packet_dispatch (pack, port, mode);
	} else if (mode) {
		if (st.network & NET_NOBCAST)
			return (1);
		pack->address = 0;	/* broadcast */
		for (port = 0; port < nnetports; ++port) {
			if (2 == mode && !netports[port].nplayers)
				continue;
			pack->netport = port;
			ret |= packet_dispatch (pack, port, mode);
		}
	} else
		ret |= packet_dispatch (pack, pack->netport, mode);
	return (ret);
}
