/* --------------------------------- oviewer.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* The special 'viewer' object is used in the external view modes.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"
#include "hud.h"

static SHAPE shape_viewer = {
	0,
	0,
	0,
	1L,		/* weight */
	0		/* drag */
};

static int FAR
init_viewer (BODY *b)
{
	if (shape_read (&shape_viewer, "viewer.vxx"))
		return (1);
	return (0);
}

static void FAR
term_viewer (BODY *b)
{
	shape_free (shape_viewer.v);
}

static int FAR
create_viewer (OBJECT *p)
{
	if (!CV)
		return (1);

	p->color = st.white;
	p->time = FOREVER;
	p->owner = CV;
	p->ownerid = CV->id;

	Vcopy (p->a, CV->a);
	Mcopy (p->T, CV->T);
	LVcopy (p->R, CV->R);
	return (0);
}

static void FAR
dynamics_viewer (OBJECT *p, int interval)
{
	OBJECT		*owner, *target;
	int		mode, f;
	long		t;
	ANGLE		a;
	VECT		V, V1;
	AVECT		da;
	MAT		M;
	LVECT		*R;
	VIEWPORT	*ovp;

	if ((owner = p->owner) != CV || owner->id != CV->id) {
		owner = CV;
		p->owner = CV;
		p->ownerid = CV->id;
		Vcopy (p->a, CV->a);
		Mcopy (p->T, CV->T);
		LVcopy (p->R, CV->R);
	}
	if (!(ovp = owner->viewport))
		ovp = CP;
	mode = p->misc[0];

	switch (mode) {
	case HDD_MAP:					/* map */
	case HDD_RADAR:					/* new map */
		p->R[X] = owner->R[X];
		p->R[Y] = owner->R[Y];
		p->R[Z] = 48000L*VONE;
		p->a[X] = -D90;
		p->a[Y] = 0;
		if (HDD_RADAR == mode)
			p->a[Z] = owner->a[Z];
		else
			p->a[Z] = 0;

		Mobj (p);
		break;
	case HDD_GAZE:
		f = COS(CP->rotx);
		f = fmul (f, 50*VONE);
		t = owner->R[X] - fmul (f, SIN(CP->rotz));
		p->R[X] += (t - p->R[X])/1;

		t = owner->R[Y] - fmul (f, COS(CP->rotz));
		p->R[Y] += (t - p->R[Y])/1;

		f = SIN(CP->rotx);
		f = fmul (f, 50*VONE);
		t = owner->R[Z] + f;
		p->R[Z] += (t - p->R[Z])/1;

		p->a[X] = 0;
		p->a[Y] = 0;
		p->a[Z] = 0;

		Mident (p->T);
		break;
	case HDD_CHASE:					/* chase orig */
	case HDD_FOLLOW:				/* chase orig, Y=0 */
		f = 50*VONE + muldiv (owner->speed, 500*VONE, VMAX);
		if (owner->speed)
			Vmuldiv (V, owner->V, f, owner->speed);
		else
			Vscale (V, owner->T[Y], f);
		Vsub (p->R, owner->R, V);

		p->a[X] += (owner->a[X] - p->a[X])/4;
		if (HDD_CHASE == mode)
			p->a[Y] += (owner->a[Y] - p->a[Y])/8;
		else
			p->a[Y] = 0;
		p->a[Z] += (owner->a[Z] - p->a[Z])/4;
		Mobj (p);
		break;
	case HDD_TARGET:			/* focus on target */
	case HDD_PAN:				/* panning target */
		LVcopy (p->R, owner->R);
		Vcopy (p->V, owner->V);
		p->da[X] = p->da[Y] = p->da[Z] = 0;
		if (((target = EE(owner)->target)) &&
						target->id == EE(owner)->tid)
			R = &target->R;
		else if (EE(owner)->hud2 & HUD_ILS)
			R = &ils[EE(owner)->ils-1].R;
		else {
			Vcopy (p->a, owner->a);
			Mcopy (p->T, owner->T);
			break;
		}
		V[X] = (int)(((*R)[X] - owner->R[X])/VONE);
		V[Y] = (int)(((*R)[Y] - owner->R[Y])/VONE);
		V[Z] = (int)(((*R)[Z] - owner->R[Z])/VONE);
		if (HDD_PAN == mode) {
#if 1
			Mxpose (p->T);
			VMmul (V1, V, p->T);
			Mxpose (p->T);

			f = TADJ (VD180)*VONE;
			a = ATAN (V1[Z], V1[Y]);
			if (a > f)
				da[X] = f;
			else if (a < -f)
				da[X] = -f;
			else
				da[X] = a;
			a = -ATAN (V1[X], V1[Y]);
			if (a > f)
				da[Z] = f;
			else if (a < -f)
				da[Z] = -f;
			else
				da[Z] = a;
			da[Y] = 0;

			Myxz (M, da);
			Mmul (M, p->T);
			Mangles1 (M, p->a);
			p->a[Y] = 0;
#endif
#if 0
			dampen (&p->a[Z], -ATAN (V[X], V[Y]), 4);
			f = ihypot2d (V[X], V[Y]);
			dampen (&p->a[X], ATAN (V[Z], f), 4);
			p->a[Y] = 0;
#endif
#if 0
			i = p->a[Z];
			dampen (&i, -ATAN (V[X], V[Y]), 4);
			p->a[Z] = i;

			f = ihypot2d (V[X], V[Y]);
			i = p->a[X];
			dampen (&i, ATAN (V[Z], f), 4);
			p->a[X] = i;

			if (iabs(p->a[X]) > D90) {
				p->a[X] = D180 - p->a[X];
				p->a[Z] = D180 + p->a[Z];
			}
			p->a[Y] = 0;
#endif
#if 0
			da[X] = TADJ(owner->da[X])*VONE;
			da[Y] = TADJ(owner->da[Y])*VONE;
			da[Z] = TADJ(owner->da[Z])*VONE;
			Myxz (M, da);
			Mmul (M, p->T);
			Mangles1 (M, p->a);

			i = p->a[Z];
			dampen (&i, -ATAN (V[X], V[Y]), 4);
			p->a[Z] = i;
			f = ihypot2d (V[X], V[Y]);
			i = p->a[X];
			dampen (&i, ATAN (V[Z], f), 4);
			p->a[X] = i;
			if (iabs(p->a[X]) > D90) {
				p->a[X] = D180 - p->a[X];
				p->a[Y] = D180 + p->a[Y];
			}
			i = p->a[Y];
			dampen (&i, 0, 4);
			p->a[Y] = i;
#endif
		} else {
			p->a[Z] = -ATAN (V[X], V[Y]);
			f = ihypot2d (V[X], V[Y]);
			p->a[X] = ATAN (V[Z], f);
			p->a[Y] = 0;
		}
		Mobj (p);
		break;
	case HDD_REAR:
#if 0
		Vcopy (p->a, owner->a);
		Mcopy (p->T, owner->T);
		LVcopy (p->R, owner->R);
		p->viewport->rotz = D180;
#else
		LVcopy (p->R, owner->R);
		p->a[X] = -owner->a[X];
		p->a[Y] = -owner->a[Y];
		p->a[Z] = D180+owner->a[Z];
		Mobj (p);
#endif
		break;
	case HDD_RIGHT:
	case HDD_LEFT:
		if (p->viewport->zoom != ovp->zoom)
			zoom (p->viewport, ovp->zoom-p->viewport->zoom);
		a = ATAN (ovp->maxx, ovp->z);
		if (HDD_RIGHT == mode)
			p->viewport->rotz = ovp->rotz + 2*a;
		else
			p->viewport->rotz = ovp->rotz - 2*a;
		goto def_case;
	case HDD_FRONT:
	default:
def_case:
		Vcopy (p->a, owner->a);
		Mcopy (p->T, owner->T);
		LVcopy (p->R, owner->R);
		break;
	}
}

BODY BoViewer = {
	0,
	0,
	"VIEWER",
	&shape_viewer,
	init_viewer,
	term_viewer,
	create_viewer,
	body_delete,
	dynamics_viewer,
	body_hit
};
