/* --------------------------------- pitch.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* paint the Head Up Display: pitch ladder
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "fly.h"
#include "hud.h"

/* Pendulum ticks:	  5      15     30     45 */
static int	Psin[] = {1428,  4240,  8192,  11585};
static int	Pcos[] = {16322, 15827, 14189, 11585};

extern void FAR
show_pitch (VIEW *view, OBJECT *p, int sx, int sy, int maxx, int maxy,
	int orgx, int orgy, int ttx, int tty, int tx, int ty, int ss,
	int shifty, int type)
{
	int	hud1, fa18, f16, f156, fx, gear;
	int	x, y, x0, y0, ex, tt;
	int	i, j, xt, yt, dx, dy, dd, rev, ndash;
	int	sroll, croll, pitch, xroll, yroll, xnum, ynum, xslant, yslant;
	int	xgap, ygap, xtip, ytip, l, px, py, n, nn, ldegrees;
    	long	lx, ly, lz;
	ANGLE	langles, hudarea, a;
	VECT	RR;
	OBJECT	*target;

	hud1 = EE(p)->hud1;
	gear = EE(p)->equip & EQ_GEAR;
	i = hud1 & HUD_TYPES;
	fa18 = i == HUD_FA18;
	f16  = i == HUD_F16;
	f156  = f16 || i == HUD_F15;
	fx = fa18 || f156;

/*
 * Show ladder.
*/
	Vcopy (RR, EE(p)->v);

	sroll = SIN(p->a[Y]);
	croll = COS(p->a[Y]);
/*
 * Set ladder center on vv or ww as needed.
*/
	if (!(EE(p)->hud & HUD_VV) || RR[Y] <= 0)
		px = py = 0;
	else {
		tt = fmul (RR[X], croll) + fmul (RR[Z], sroll);
		tt = muldiv (tt, VVPERIOD-VVDELAY, VVPERIOD);
		RR[X] = fmul (tt, croll);
		RR[Z] = -fmul (tt, sroll);
		screen_coords (view, RR);

		if ((Uint)RR[Y] <= iabs (RR[X])/2 ||
		    (Uint)RR[Y] <= iabs (RR[Z])/2)
			goto noladder;
		px = muldiv(RR[X], maxx, RR[Y]);
		py = muldiv(RR[Z], maxy, RR[Y]);
	}
/*
 * Choose ladder pitch.
*/
	if (HDD_HUD == type) {
		hudarea = DEG2ANG (EE(p)->hudarea);
		y = muldiv (view->viewport->z, SIN (hudarea), COS (hudarea));
		y += fmul (y, EE(p)->misc[17]);
		if (y > view->viewport->maxy) {
			y = fmul (view->viewport->z, FONE + EE(p)->misc[17]);
			hudarea = ATAN (view->viewport->maxy, y);
		}
		langles = hudarea;
	} else {
		y = muldiv (view->viewport->maxy, sy, maxy);
		langles = ATAN (y, view->viewport->z);
		hudarea = 0;	/* avoid compiler warning */
	}
	ldegrees = ANG2DEG00 (langles);

	if (ldegrees < 300)
		ldegrees = 1;		/* degrees per step */
	else if (ldegrees < 500) 	/* must be a factor of 90 */
		ldegrees = 2;
	else if (ldegrees < 1500)
		ldegrees = 5;
	else if (ldegrees < 2300)
		ldegrees = 10;
	else if (ldegrees < 4500)
		ldegrees = 15;
	else
		ldegrees = 30;

	langles = DEG2ANG (ldegrees);	/* angle of step */

	if (HDD_HUD == type) {
		x0 = muldiv (sx, langles, hudarea);
		y0 = muldiv (sy, langles, hudarea);
	} else {
		x0 = muldiv (maxx, langles, ATAN (view->viewport->maxx,
				view->viewport->z));
		y0 = muldiv (maxy, langles, ATAN (view->viewport->maxy,
				view->viewport->z));
	}

	x0 = fmul (sroll, x0);
	y0 = fmul (croll, y0);

	dd = num_size (9L, ss);
	if (fx) {
		xnum = fmul (croll, 2*dd+2);
		ynum = fmul (sroll, 2*dd+2);
	} else {
		xnum = fmul (croll, ss);
		ynum = fmul (sroll, ss);
	}

	xt = fmul (sroll, ss);
	yt = fmul (croll, ss);
	xtip = fa18 ? xt : xt/2;	/* ladder wing-tip size */
	ytip = fa18 ? yt : yt/2;

	sroll = fmul (sroll, sy);	/* convert to hud pixels */
	croll = fmul (croll, sx);
/*
 * pull-up cue
*/
	if (WE_MK82 != EE(p)->weapon)
		goto no_cue;

	if (!(target = EE(p)->target) || target->id != EE(p)->tid)
		goto no_cue;

	lx = (target->R[X] - p->R[X])/VONE;
	ly = (target->R[Y] - p->R[Y])/VONE;
	lz = (target->R[Z] + 500L*VONE - p->R[Z])/VONE;
	if (labs(lx) > 25000L || labs(ly) > 25000L || labs(lz) > 25000L)
		goto no_cue;			/* too far out */

	tt = ihypot2d ((int)lx, (int)ly);	/* dist. to target */
	a = ATAN ((int)lz, tt) - p->a[X];

	xgap  = fmul (croll, PULLUPCUEGAP);
	ygap  = fmul (sroll, PULLUPCUEGAP);
	xroll = fmul (croll, PULLUPCUE);
	yroll = fmul (sroll, PULLUPCUE);
	xslant = fmul (croll, PULLUPCUETIP);
	yslant = fmul (sroll, PULLUPCUETIP);
	x = orgx + px - muldiv (x0, a, langles);
	y = orgy + py - muldiv (y0, a, langles);

	add_segment (x-xgap, y+ygap, x-xroll, y+yroll, st.hfg, orgx,
		orgy, sx, sy, shifty);
	add_segment (x+xgap, y-ygap, x+xroll, y-yroll, st.hfg, orgx,
		orgy, sx, sy, shifty);

	add_segment (x-xroll, y+yroll, x-xslant-xt/2, y+yslant-yt/2, st.hfg,
		orgx,orgy, sx, sy, shifty);
	add_segment (x+xroll, y-yroll, x+xslant-xt/2, y-yslant-yt/2, st.hfg,
		orgx, orgy, sx, sy, shifty);
no_cue:	;

	tt = f156 ? F15GAP : (fx ? F18GAP : FONE/32*5);
	xgap  = fmul (croll, tt);	/* gap pixels */
	ygap  = fmul (sroll, tt);

     	ndash = f16 ? 12 : 10;

	xslant = 0;
	yslant = 0;

	a = p->a[X];
	l = ANG2DEG (a);		/* degrees */
	if (a >= 0) {
		pitch = (int)(a % langles);
		ex = (int)(a / langles);
	} else {
		a = -a;
		pitch = langles - (int)(a % langles);
		ex = -(int)(a / langles + 1);
	}
	pitch *= (90/ldegrees);
	px += fmul (pitch, x0);
	py += fmul (pitch, y0);

	for (i = -3; i <= 3; ++i) {
		x = px - i*x0;
		y = py - i*y0;

		n = (ex + i) * ldegrees;
		nn = n-l;

		if (n > 90) {
			n = 180 - n;
			rev = 1;
		} else if (n < -90) {
			n = -180 - n;
			rev = 1;
		} else
			rev = 0;

		if (f156)
			tt = n ? F15STEP : F15STEP0;
		else
			tt = (fx)
				? (n ? F18STEP : (gear ? F18STEPG : F18STEP0))
				: (n ? FONE/32*12 : FONE/32*18); /* Classic */
		xroll = fmul (croll, tt);
		yroll = fmul (sroll, tt);

		nn = abs(n);
		if (fa18 && 90 == nn){
			dx = 2*tx;
			dy = 2*ty;
			if (iabs(x) < (Uint)(sx-dx) &&
			    y < sy-dy+shifty && y > -sy+dy+shifty) {
				x += orgx;
				y += orgy;
				add_5op (T_ELLIPSE, x, y, dx, dy, st.hfg);
				if (n < 0) {
					tt = SIN (D90/2);
					dx = fmul (tt, dx);
					dy = fmul (tt, dy);
					add_line (x-dx, y+dy, T_MOVE);
					add_line (x+dx, y-dy, st.hfg);
					add_line (x-dx, y-dy, T_MOVE);
					add_line (x+dx, y+dy, st.hfg);
				}
			}
			continue;
		}

		x += orgx;
		y += orgy;

		if (rev) {
			xt = -xt;
			yt = -yt;
			xtip = -xtip;
			ytip = -ytip;
			xnum = -xnum;
			ynum = -ynum;
			xroll = -xroll;
			yroll = -yroll;
			xgap = -xgap;
			ygap = -ygap;
		}

		if (fa18) {
			xslant = muldiv (xt, nn, 90/2);
			yslant = muldiv (yt, nn, 90/2);
		}

		if (n < 0) {
			add_segment (x-xroll-xtip, y+yroll-ytip, x-xroll,
				y+yroll, st.hfg, orgx, orgy, sx, sy, shifty);
			add_segment (x+xroll-xtip, y-yroll-ytip, x+xroll,
				y-yroll, st.hfg, orgx, orgy, sx, sy, shifty);

		     	dx = xroll - xgap;
		     	dy = yroll - ygap;
			for (j = 0; j < ndash; j += 2) {
				int	dx0, dy0, dx1, dy1, xs0, ys0, xs1, ys1;

			     	xs0 = x - muldiv (xslant, j, ndash);
			     	ys0 = y - muldiv (yslant, j, ndash);
			     	dx0 = xroll - muldiv (dx, j, ndash);
			     	dy0 = yroll - muldiv (dy, j, ndash);
			     	xs1 = x - muldiv (xslant, j+1, ndash);
			     	ys1 = y - muldiv (yslant, j+1, ndash);
			     	dx1 = xroll - muldiv (dx, j+1, ndash);
			     	dy1 = yroll - muldiv (dy, j+1, ndash);
				add_segment (xs0-dx0, ys0+dy0, xs1-dx1,
					ys1+dy1, st.hfg, orgx, orgy, sx, sy,
					shifty);
				add_segment (xs0+dx0, ys0-dy0, xs1+dx1,
					ys1-dy1, st.hfg, orgx, orgy, sx, sy,
					shifty);
			}
		} else {
			if (n || fa18) {
				add_segment (x-xroll+xtip, y+yroll+ytip,
					x-xroll, y+yroll, st.hfg, orgx, orgy,
					sx, sy, shifty);
				add_segment (x+xroll+xtip, y-yroll+ytip,
					x+xroll, y-yroll, st.hfg, orgx, orgy,
					sx, sy, shifty);
			}

			add_segment (x-xroll, y+yroll, x-xgap+xslant,
				y+ygap+yslant, st.hfg, orgx, orgy, sx, sy,
				shifty);
			add_segment (x+xroll, y-yroll, x+xgap+xslant,
				y-ygap+yslant, st.hfg, orgx, orgy, sx, sy,
				shifty);
		}

		if (n) {
			int	xl, yl, xr, yr;

			xl = x-xroll;
			yl = y+yroll;
			xr = x+xroll;
			yr = y-yroll;

			if (fa18) {		/* rotated numerals */
				stroke_angle (rev ? D180+p->a[Y] : p->a[Y]);
				if (n >= 0) {
					if (n < 10) {
						xl += -xnum/2+xt;
						yl +=  ynum/2+yt;
					} else {
						xl += -xnum+xt;
						yl +=  ynum+yt;
					}
					xr +=  xnum/4+xt;
					yr += -ynum/4+yt;
				} else {
					if (n > -10) {
						xl += -xnum/2;
						yl +=  ynum/2;
					} else {
						xl += -xnum;
						yl +=  ynum;
					}
					xr +=  xnum/4;
					yr += -ynum/4;
				}
			} else if (f156) {	/* rotated numerals */
				stroke_angle (rev ? D180+p->a[Y] : p->a[Y]);
				if (n >= 0) {
					if (n < 10) {
						xl += -xnum/2+xt/2;
						yl +=  ynum/2+yt/2;
					} else {
						xl += -xnum+xt/2;
						yl +=  ynum+yt/2;
					}
					xr +=  xnum/4+xt/2;
					yr += -ynum/4+yt/2;
				} else {
					if (n > -10) {
						xl += -xnum/2+xt/2;
						yl +=  ynum/2+yt/2;
					} else {
						xl += -xnum+xt/2;
						yl +=  ynum+yt/2;
					}
					xr +=  xnum/4+xt/2;
					yr += -ynum/4+yt/2;
				}
			} else {		/* upright numerals */
				dx = (nn >= 10) ? dd : dd/2;
				xl += -xnum-dx;
				yl +=  ynum+ss/2;
				xr +=  xnum-dx;
				yr += -ynum+ss/2;
			}
			show_num (xl, yl, (long)nn, ss, st.hfg, orgx, orgy,
				sx, sy, shifty);
			show_num (xr, yr, (long)nn, ss, st.hfg, orgx, orgy,
				sx, sy, shifty);
		}

		if (rev) {
			xt = -xt;
			yt = -yt;
			xtip = -xtip;
			ytip = -ytip;
			xnum = -xnum;
			ynum = -ynum;
			xgap = -xgap;
			ygap = -ygap;
		}
	}
	if (fx)
		stroke_angle (0);
noladder:

/*
 * Show pendulum.
*/
	if (fa18 && (hud1 & HUD_PENDULUM)) {
		px = fmul (sx, PENSIZE);
		py = fmul (sy, PENSIZE);
		add_line (orgx, orgy+py-py/32, T_MOVE);
		add_line (orgx, orgy+py,       st.hfg);
		for (i = 0; i < 4; ++i) {
			dx = Psin[i];
			dy = Pcos[i];
			dx = fmul (px, dx);
			dy = fmul (py, dy);
			x = dx/32;
			y = dy/32;
			dx -= x;
			dy -= y;
			if (0 == i) {
				x = (x+1)/2;
				y = (y+1)/2;
			}
			add_line (orgx+dx,   orgy+dy,   T_MOVE); /* +ve */
			add_line (orgx+dx+x, orgy+dy+y, st.hfg);
			add_line (orgx-dx,   orgy+dy,   T_MOVE); /* -ve */
			add_line (orgx-dx-x, orgy+dy+y, st.hfg);
		}
		a = p->a[Y];
		if (a > DEG2ANG(50)) {
			a = DEG2ANG(50);
			j = 1;
		} else if (a < -DEG2ANG(50)) {
			a = -DEG2ANG(50);
			j = 1;
		} else
			j = 0;
		if (!j || (((int)st.present)&0x0080)) {
			x  = SIN (a);
			y  = COS (a);
			dx = fmul (px, x);	/* arrow point */
			dy = fmul (py, y);
			xt = fmul (px, y)/64;	/* arrow base */
			yt = fmul (py, x)/64;
			px = dx/32;
			py = dy/32;
			dx += orgx-px;
			dy += orgy-py;
			add_line (dx-px,    dy-py,    T_MOVE);
			add_line (dx,       dy,       st.hfg);
			add_line (dx-px+xt, dy-py-yt, st.hfg);
			add_line (dx-px-xt, dy-py+yt, st.hfg);
			add_line (dx,       dy,       st.hfg);
		}
	}
}
