/* --------------------------------- pointer.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Pointing devices manager.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "fly.h"

extern struct PtrDriver *PtrDrivers[];

extern int FAR
pointers_init (void)
{
	return (0);
}

extern void FAR
pointers_term (void)
{
	return;
}

extern POINTER * FAR
pointer_select (char *pname)
{
	int	i, c;
	char	*p;
	int	options[NOPTS];
	POINTER	*ptr;

	memset (options, 0, sizeof (options));
	options[0] = 1;		/* +ve x */
	options[1] = 0;		/* x axis is channel 0 */
	options[2] = 1;		/* +ve y */
	options[3] = 1;		/* y axis is channel 1 */
	options[4] = 5;		/* right button: fire */
	options[5] = 18;	/* left  button: unlock */
	options[6] = 0;		/* mouse x sensitivity */
	options[7] = 0;		/* mouse y sensitivity */
	options[8] = 1;		/* read joystick once */

	if (!pname) {
		i = 0;			/* use default */
		goto ret;
	}

	p = strchr (pname, '=');
	if (p) {		/* "+x+ybbss=" */
		if (pname[0] == 'n')		/* x direction */
			options[0] = -1;
		else if (pname[0] == 'p')
			options[0] = 1;
		else
			return (0);

		if ((options[1] = opt36 (pname[1])) < 0) /* x select */
			return (0);

		if (pname[2] == 'n')		/* y direction */
			options[2] = -1;
		else if (pname[2] == 'p')
			options[2] = 1;
		else
			return (0);

		if ((options[3] = opt36 (pname[3])) < 0) /* y select */
			return (0);

		for (i = 4; i < NOPTS && (c = pname[i]) != '='; ++i) {
			if ((c = opt36 (c)) < 0)
				return (0);	/* too bad */
			options[i] = c;
		}
		if (pname[i] != '=')
			return (0);		/* too many options */

		pname = p+1;
	}
	if (0 == options[8])
		options[8] = 1;		/* read joystick once */

	for (i = 0; PtrDrivers[i]; ++i)
		if (!stricmp (PtrDrivers[i]->name, pname))
			goto ret;
	return (0);

ret:
	if (!NEW (ptr))
		return (0);
	ptr->control = PtrDrivers[i];
	ptr->name    = PtrDrivers[i]->name;
	memcpy (ptr->opt, options, sizeof (ptr->opt));
	return (ptr);
}

extern POINTER * FAR
pointer_release (POINTER *ptr)
{
	ptr->control->Term (ptr);
	DEL (ptr);
	return (0);
}

extern void FAR
std_key (POINTER *p, int key)
{
	switch (key) {
	case KF_POWER_UP:
		p->b[0] += 5;
		break;

	case KF_POWER_DOWN:
		p->b[1] += 5;
		break;

	case KF_POWER_0:
		++p->b[4];
		break;

	case KF_POWER_100:
		++p->b[16];
		break;

	case KF_POWER_AB:
		++p->b[17];
		break;

	case KF_LEVEL:
		++p->b[2];
		break;

	case KF_ORIGIN:
		++p->b[3];
		break;

	case KF_FIRE:
		++p->b[5];
		break;

	case KF_RESET_ROLL:
		++p->b[12];
		break;

	case KF_STABLE:
		++p->b[8];
		break;

	case KF_FRUDLEFT:
		++p->b[6];		/* rudder left */
		break;
	case KF_FRUDRITE:
		++p->b[7];		/* rudder right */
		break;
	case KF_FRUDCNTR:
		++p->b[9];		/* rudder center */
		break;

	case ']':			/* flaps */
		++p->b[10];
		break;
	case '[':
		--p->b[10];
		break;

	case '}':			/* spoilers */
		++p->b[11];
		break;
	case '{':
		--p->b[11];
		break;

	case '+':			/* air (speed) brakes */
		++p->b[13];
		break;

	case 'b':
		++p->b[14];		/* wheels brakes */
		break;

	case 'g':
		++p->b[15];		/* landing gear */
		break;

	case ' ':
		++p->b[18];		/* release lock */
		break;
	}
}

/*
 * Select Pointing Device
*/

extern int FAR
menu_ptrs (void)
{
	MENU	*MenuPtr;
	int	sel, i, n, nEntries, EntrySize;
	short	flags;
	char	*oldptr, newptr[NOPTS+1+32];
	POINTER	*ptr;

	for (nEntries = 0; PtrDrivers[nEntries]; ++nEntries);
		;
	EntrySize = 20;

	n = (nEntries+1) * sizeof (*MenuPtr);
	if (!(MenuPtr = (MENU *) memory_alloc (n)))
		return (1);

	sel = MENU_FAILED;
	for (i = 0; i < nEntries; ++i)
		if (!(MenuPtr[i].text = (char *) memory_alloc (EntrySize)))
			goto ret;

	if ((oldptr = strchr (st.pname, '=')))
		++oldptr;
	else
		oldptr = st.pname;
	sel = 0;
	for (i = 0; i < nEntries; ++i) {
		MenuPtr[i].letter = (Uchar)menuch[i];
		strcpy (MenuPtr[i].text, PtrDrivers[i]->name);
		if (!stricmp (PtrDrivers[i]->name, oldptr))
			sel = i;
	}

	sel = menu_open (MenuPtr, sel);

	oldptr = st.pname;

	switch (sel) {
	case MENU_ABORTED:
	case MENU_FAILED:
		break;
	default:
		getstr ("pointer options:", newptr, NOPTS+1);
		if ('\0' != newptr[0])
			strcat (newptr, "=");
		strcat (newptr, PtrDrivers[sel]->name);

		flags = st.flags;
		st.flags |= SF_SIMULATING;
		if ((ptr = CV->pointer))
			CV->pointer = pointer_release (ptr);
		for (;;) {
			ptr = pointer_select (newptr);
			if (ptr && !ptr->control->Init (ptr)) {
				CV->pointer = ptr;
				st.pname = xfree (st.pname);
				st.pname = strdup (newptr);
				break;
			}
			MsgEPrintf (-100, "pointer init failed");
			ptr = pointer_select (oldptr);
			if (ptr && !ptr->control->Init (ptr)) {
				CV->pointer = ptr;
				break;
			}
			MsgEPrintf (-100, "old pointer init failed");
			if (oldptr) {
				ptr = pointer_select (NULL);
				if (ptr && !ptr->control->Init (ptr)) {
					CV->pointer = ptr;
					st.pname = xfree (st.pname);
					st.pname = strdup (ptr->name);
					break;
				}
			}
			LogPrintf ("default pointer init failed\n");
			die ();
		}
		st.flags = flags;
		break;
	}

ret:
	for (i = 0; i < nEntries; ++i)
		if (MenuPtr[i].text)
			memory_free (MenuPtr[i].text, EntrySize);

	memory_free (MenuPtr, n);

	if (MENU_FAILED == sel)
		return (1);

	menu_close ();
	return (0);
}
