/* -------------------------------- show.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* A 3D graphics engine.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"

#if 0
/*
 * This one tries a different projection. It produces a more uniform
 * distortion in wide angle. The method is to project the world onto a sphere
 * and then use the latitude/longitude as the x/y on the screen. Perspective
 * projection shows objects larger on the edges that in the center, this one
 * shows equal size objects (same angle of view) with equal size.
 * The problem is that straight lines show as arcs!
*/
#define	PROJ(x,pr,z) \
	(projtype ? fmul((pr)*2,ATAN((x),(z))) : muldiv((x),(pr),(z)))
#else
#define	PROJ(x,pr,z)	muldiv((x),(pr),(z))
#endif

#define PROJPT(x,y,z,org,pr) \
	((z) ? add_line (org[X]+PROJ(x,pr[X],z),		\
				org[Y]-PROJ(y,pr[Y],z), T_MOVE)	\
	     : add_line (org[X], org[Y], T_MOVE))

#define PROJMV(x,y,z,org,pr,c) \
	((z) ? add_line (org[X]+PROJ(x,pr[X],z),		\
				org[Y]-PROJ(y,pr[Y],z), c)	\
	     : add_line (org[X], org[Y], c))

#define GETALPHA(a,b)	(tt2 = B[Z]/2 + b/2 + (tt1 =  a/2 - A[Z]/2))

#define CLIP(a,b)	(a + 2*muldiv(b/2-a/2,tt1,tt2))

static VECT	NEAR V1 = {0}, NEAR A = {0}, NEAR B = {0};

static int FAR
adj (void)
{
	register int	tt1, tt2, tt3;

	/*
	 * A straight forward clipping routine. Clip againt the right, left
	 * top and bottom in order.
	 *
	 */

	if (V1[Z] < V1[Y]) {
		GETALPHA (A[Y], -B[Y]);
		if (tt2  && (tt3 = CLIP (A[Z], B[Z])) > 0) {
			V1[Y] = V1[Z] = tt3;
			V1[X] = CLIP (A[X], B[X]);
		}
	}
	if (V1[Z] < -V1[Y]) {
		GETALPHA (-A[Y], B[Y]);
		if (tt2  && (tt3 = CLIP (A[Z], B[Z])) > 0) {
			V1[Y] = -(V1[Z] = tt3);
			V1[X] = CLIP (A[X], B[X]);
		}
	}

	if (V1[Z] < V1[X]) {
		GETALPHA (A[X], -B[X]);
		if (tt2  && (tt3 = CLIP (A[Z], B[Z])) > 0) {
			V1[X] = V1[Z] = tt3;
			V1[Y] = CLIP (A[Y], B[Y]);
		}
	}
	if (V1[Z] < -V1[X]) {
		GETALPHA (-A[X], B[X]);
		if (tt2  && (tt3 = CLIP (A[Z], B[Z])) > 0) {
			V1[X] = -(V1[Z] = tt3);
			V1[Y] = CLIP (A[Y], B[Y]);
		}
	}

	if (V1[Z] < V1[X] || V1[Z] < -V1[X] ||
	    V1[Z] < V1[Y] || V1[Z] < -V1[Y])
		return (1);
	return (0);
}

static void FAR
show_line (VERTEX *v, MAT TT, VECT R, int SH[2], int PR[2], int ORG[2],
	int c)
{
	register int	xxxx;
	static int	yyyy, prev = 0;

	++st.stats[15];

	/*
	 * v->V is in object coordinates
	 */

	VMmul (A, v->V, TT);		/* rotate */


	/*
	 * A is in viewer's coordinates, but at origin.
	 */

	Vinc (A, R);			/* translate (R already rotated) */

	/*
	 * A is now in viewer coordinates, in position.
	 *
	 * We now swap y and z in the viewer's coordinate system:
	 * x was 'right' (still is)
	 * y was 'forward' (now is 'up', so the screen is x-y)
	 * z was 'up' (now is 'forward', perpendicular to the screen)
	 *
	 */

	xxxx = A[Z];	/* xxxx used as temp */
	A[Z] = A[Y];
	A[Y] = xxxx;

	/*
	 * shift the x and y to compensate for viewport offset.
	 * SH is in viewers coordinates.
	 * This is not properly done. SH[] is always zero for now.
	 * Actualy, the "dynamic" scaling for truncation control makes this
	 * shift WRONG!
	 *
	 */

	A[X] -= SH[X];
	A[Y] -= SH[Y];

	/*
	 * xxxx is clipping pattern:
	 *	0x01: x >  z (too far right)
	 *	0x02: x < -z (too far left)
	 *	0x04: y >  z (too far up)
	 *	0x08: y < -z (too far down)
	 *	0x10: z <  0 (behind)
	 *
	 */

 	xxxx = 0;
	if (A[X] > A[Z])
		xxxx |= 0x01;
	if (A[X] < -A[Z])
		xxxx |= 0x02;
	if (A[Y] > A[Z])
		xxxx |= 0x04;
	if (A[Y] < -A[Z])
		xxxx |= 0x08;
	if (A[Z] < 0)
		xxxx |= 0x10;

	Vcopy (V1, A);

	/*
	 * Use the bit pattern of the current point and the previous one to
	 * attempt fast rejects/accepts and avoid un-necessary clipping.
	 *
	 * 'V_MOVE' means the segment into this point is not to be drawn.
	 *
	 * 'prev' is true if the last point was not drawn (which means the
	 * graphics device is not at the current position).
	 *
	 * adj() clips one endpoint.
	 *
	 * PROJMV() projects a point and draw a line to it.
	 *
	 * PROJPT() projects a point and issues command to move there without
	 * drawing (just a re-positioning).
	 *
	 */

	if (V_MOVE == v->flags || V_DUP == v->flags)
		goto noline;
	if (xxxx == yyyy) {
		if (xxxx)
			goto reject;
		else
			goto same;
	}
	if (xxxx & yyyy)
		goto reject;
	if (0 == prev)
		goto newout;
	if (0 == yyyy)
		goto oldin;
	if (0 == xxxx)
		goto newin;
	/*
	 * The case is undecided so one clipping is attempted. If it fails
	 * then we reject this segment. This is the only time a failure is
	 * expected. In the other cases, a failure of adj() means that an
	 * unexpected problem prevented it from working - usualy a result
	 * of lost resolution due to truncation.
	 *
	 */

	++st.stats[13];

	if (adj ())		/* a trial clipping */
		goto noline;
	PROJPT (V1[X], V1[Y], V1[Z], ORG, PR);
	Vcopy (V1, B);

	if (0) {
newout:
		++st.stats[12];
	}
	if (adj ())		/* should not fail */
		goto noline;
	PROJMV (V1[X], V1[Y], V1[Z], ORG, PR, c);
	if (0) {
reject:
		++st.stats[10];
	}
noline:
	prev = 1;
	goto ret;

oldin:
	PROJPT (B[X], B[Y], B[Z], ORG, PR);
	goto newout;
newin:
	++st.stats[12];
	Vcopy (V1, B);

	if (adj ())		/* should not fail */
		goto noline;
	PROJPT (V1[X], V1[Y], V1[Z], ORG, PR);
	goto newin1;
same:
	++st.stats[11];
	if (prev) {
		PROJPT (B[X], B[Y], B[Z], ORG, PR);
	}
newin1:
	PROJMV (A[X], A[Y], A[Z], ORG, PR, c);
	prev = 0;
ret:
	yyyy = xxxx;
	Vcopy (B, A);
}

static VERTEX shape_dot[] = {
	{{0, 0, 0}, V_MOVE},
	{{0, 0, 0}, V_DRAW},
	{{0, 0, 0}, 0}
};

static void FAR
show_object (OBJECT *obj, MAT VT, LVECT VR, int SH[2], int PR[2], int ORG[2],
	int frame, long minextent, LVECT OR, VECT RR)
{
	register VERTEX	*v;
	VECT		P, R;
	MAT		TT;
	LVECT		LP;
	VERTEX		V[1];
	long		l, dist, extent;
	int		scale, too_far, too_near, obj_size, depth, color;
	int		t, b;

	/*
	 * Get vector from viewr to object.
	 *
	 */

	if (obj)
		Vsub (LP, obj->R, VR);
	else
		Vsub (LP, OR, VR);

	/*
	 * If the object is too far, show only a dot. This enables the
	 * use of 16-bit arithmetic in a 32-bit world. If it is much further
	 * then don't show it at all.
	 *
	 */

	dist = labs (LP[X]);
	if ((l = labs (LP[Y])) > dist)
		dist = l;
	if ((l = labs (LP[Z])) > dist)
		dist = l;

	if (obj && (Uint)(dist/4/minextent) > SH(obj)->extent)
		return;

	if (obj)
		obj_size = SH(obj)->extent;
	else
		obj_size = 1;
	extent = dist + obj_size*(long)VONE;
	if (extent == 0)
		return;

	too_near = too_far = 0;
	if (extent > 0x7fffL*VONE) {
		/*
		 * Bring far objects nearer.
		 *
		 */

		too_far = 1;
		scale = (int)(extent / (0x7fff/2));
		scale = (scale/VONE)*VONE;
		P[X] = (int)(LP[X] / scale);
		P[Y] = (int)(LP[Y] / scale);
		P[Z] = (int)(LP[Z] / scale);
		scale /= VONE;
		obj_size /= scale;
	} else if (extent <= 0x7fff/4) {
		/*
		 * Reduce truncation errors: push near points away.
		 *
		 */

		too_near = 1;
		scale = (int)((0x7fff/2) / extent);
		P[X] = ((int)LP[X]) * scale;
		P[Y] = ((int)LP[Y]) * scale;
		P[Z] = ((int)LP[Z]) * scale;
		scale *= VONE;
		obj_size *= scale;
	} else if (extent <= 0x7fffL*VONE/4) {
		/*
		 * Reduce truncation errors: push near points away.
		 *
		 */

		too_near = 1;
		scale = (int)((0x7fffL*VONE/2) / extent);
		P[X] = (int)((LP[X] * scale) / VONE);
		P[Y] = (int)((LP[Y] * scale) / VONE);
		P[Z] = (int)((LP[Z] * scale) / VONE);
		obj_size *= scale;
	} else {
		/*
		 * Shift out the fraction and move to 16-bits.
		 *
		 */

		P[X] = (int)vuscale (LP[X]);
		P[Y] = (int)vuscale (LP[Y]);
		P[Z] = (int)vuscale (LP[Z]);
		scale = 1;
	}

	/*
	 * Rotate to viewer's orientation.
	 *
	 */

	if (!obj) {
		VMmul (RR, P, VT);
		return;
	}

	VMmul (R, P, VT);	/* object's origin in viewer coordinates */

	/*
	 * Rough check for object out of view (trivial reject).
	 *
	 */

	if (R[Y] <= -obj_size)		/* all behind */
		return;
	depth = R[Y]/2 + obj_size;	/* avoid truncation */
	if (iabs(R[X]/2) > (Uint)depth || iabs(R[Z]/2) > (Uint)depth)
		return;

	/*
	 * Post conatenate Viewer's transposed matrix to the Object's.
	 *
	 */

	VMmul (TT[0], obj->T[0], VT);	/* Mmul (TT, obj->T, VT);*/
	VMmul (TT[1], obj->T[1], VT);
	VMmul (TT[2], obj->T[2], VT);

	/*
	 * Far objects shown as one dot.
	 *
	 */

	if ((Uint)(dist/minextent) > SH(obj)->extent)
		v = shape_dot;
	else
		v = SH(obj)->v;

	/*
	 * Set object color (natural or red/blue stereo).
	 *
	 */

	switch (frame) {
	default:
	case 0:
		if ((Uint)(dist*2/minextent) > SH(obj)->extent)
			color = st.gray;	/* farther than dot/2 */
		if ((Uint)(dist*4/minextent) > SH(obj)->extent)
			color = st.lgray;	/* farther than dot/4 */
		else
			color = obj->color;
		break;
	case 1:
		color = st.red;			/* left */
		break;
	case 2:
		color = st.blue;		/* right */
		break;
	}

	/*
	 * Show each line segment in turn.
	 *
	 */
#if 1
	if (SH(obj)->flags & SH_FINE) {
		if (too_near) {
			t = scale;
			b = VONE;
		} else if (too_far) {
			t = 1;
			b = scale * VONE;
		} else {
			t = 1;
			b = VONE;
		}
	} else {
		if (too_near) {
			t = scale;
			b = 1;
		} else if (too_far) {
			t = 1;
			b = scale;
		} else {
			t = 1;
			b = 1;
		}
	}
#endif
	for (; v->flags; ++v) {
#if 0
		if (too_near) {
			V->V[X] = v->V[X] * scale;
			V->V[Y] = v->V[Y] * scale;
			V->V[Z] = v->V[Z] * scale;
			V->flags = v->flags;
			show_line (V, TT, R, SH, PR, ORG, color);
		} else if (too_far) {
			V->V[X] = v->V[X] / scale;
			V->V[Y] = v->V[Y] / scale;
			V->V[Z] = v->V[Z] / scale;
			V->flags = v->flags;
			show_line (V, TT, R, SH, PR, ORG, color);
		} else
			show_line (v, TT, R, SH, PR, ORG, color);
#else
		Vmuldiv (V->V, v->V, t, b);
		V->flags = v->flags;
		show_line (V, TT, R, SH, PR, ORG, color);
#endif
	}
}

#define	TSCALE	9459		/* sqrt(1/3): cube diagonal ratio to edge */

#undef CP
#undef CW
#undef CS

#define CP	view->viewport
#define CW	view->window
#define CS	view->screen

extern void FAR
objects_show (VIEW *view, OBJECT *pov, int frame, LVECT OR, VECT RR)
{
	register int	i;
	register OBJECT	*p;
	MAT		VT;		/* rotates world to viewer */
	LVECT		R;		/* eye position */
	VECT		SC;		/* scaling */
	VECT		E, EYE;  	/* relative eye shift */
	int		SH[2];		/* viewport shift from eye, x and y */
	int		PR[2];		/* perspective ratio */
	int		ORG[2];		/* pixel origin */
	int		j;
	int		l, t;
	long		minextent;	/* minimum extent ratio to show */
	long		tt;

	if (!OR)
		tt = Tm->Milli ();
	else
		tt = 0;		/* avoid compiler warning */

	memset (st.stats+10, 0, 10*sizeof (*st.stats));

	/*
	 * This is the entry point of this module.
	 *
	 * Adjust for eye distance and viewport size so that after total
	 * scaling there is no overflow. TSCALE is sqrt(1/3) and is the
	 * minimal scaling needed due to the 3D rotation. The rest is a
	 * distortion that accounts for the pictures aspect ratio, later
	 * the program works in a square 45 degrees clip window.
	 *
	 */

	l = CP->z;				/* get minimum */
	if (l > CP->maxx)
		l = CP->maxx;
	if (l > CP->maxy)
		l = CP->maxy;

	SC[X] = muldiv (TSCALE, l, CP->maxx);	/* never divide by zero!!! */
	SC[Y] = muldiv (TSCALE, l, CP->maxy);
	SC[Z] = muldiv (TSCALE, l, CP->z);

	/*
	 * Adjust for viewer's offset to the viewport.
	 * (not quite right)
	 *
	 */

	SH[X] = fmul (CP->x, SC[X]);
	SH[Y] = fmul (CP->y, SC[Y]);

	/*
	 * Calculate paralax.
	 *
	 */

	t = muldiv (CP->z, CP->shift, CP->distz);	/* normalized */
	t = muldiv (CS->sizex, t, CP->maxx) / 2;	/* pixels */

	/*
	 * Set the perspective ratio.
	 *
	 */

	PR[X] = fmul (CW->maxx, CS->sizex - (t<0 ? -t : t)) - 1;
	PR[Y] = fmul (CW->maxy, CS->sizey) - 1;

	/*
	 * Set picture pixel origin.
	 *
	 */

	t = fmul (t, CW->maxx) + (t<0);
	ORG[X] = fmul (CW->orgx, CS->sizex) + CS->minx + t;
	ORG[Y] = fmul (CW->orgy, CS->sizey) + CS->miny;

	/*
	 * Swap y and z when changing to viewer coordinates.
	 *
	 */

	t = SC[Y];
	SC[Y] = SC[Z];
	SC[Z] = t;

	/*
	 * Transpose viewer orientation matrix.
	 * The object's T maps from the object to the world and for
	 * the viewer we need the reverse mapping.
	 *
	 */

	Mcopy (VT, pov->T);
	Mxpose (VT);

	/*
	 * Turn for viewer's head turning (and stereo cross-eye).
	 *
	 */

	if (CP->rotz)
		Mrotz (VT, CP->rotz);
	if (CP->rotx)
		Mrotx (VT, CP->rotx);
	if (CP->roty)
		Mroty (VT, CP->roty);

	/*
	 * Now do the scaling.
	 *
	 */

	for (i = 0; i < 3; ++i)		/* scale */
		for (j = 0; j < 3; ++j)
			VT[j][i] = fmul (VT[j][i], SC[i]);

	/*
	 * Find eye position.
	 *
	 */

	EYE[X] = CP->eyex + CP->shift / VONE;
	EYE[Y] = CP->eyey;
	EYE[Z] = CP->eyez;
	VMmul (E, EYE, pov->T);
	Vadd (R, pov->R, E);

	/*
	 * Calculate maximum distance for showing detail.
	 *
	 */

	minextent = CP->z * (long)CS->sizex / CP->maxx * VONE;
	if (minextent < 1)
		minextent = 1;

	/*
	 * If only object position requested, get it and quit.
	 *
	 */

	if (OR) {
		show_object (0, VT, R, SH, PR, ORG, frame, minextent, OR, RR);
		return;
	}

	/*
	 * Now show each object in turn.
	 *
	 */

	for (i = 0, p = CL; p; p = p->next) {
		if (p->flags & F_VISIBLE) {
			show_object (p, VT, R, SH, PR, ORG, frame, minextent,
					0, 0);
			if (!i--) {
				i = 20;
				sys_poll ();		/* poll frequently */
			}
		}
	}

	for (p = CO; p; p = p->next) {
		if (p == pov)			/* don't show viewer */
			continue;
		if (p->flags & F_VISIBLE) {
			show_object (p, VT, R, SH, PR, ORG, frame, minextent,
					0, 0);
			if (!i--) {
				i = 20;
				sys_poll ();		/* poll frequently */
			}
		}
	}
	st.stats[27] += Tm->Milli () - tt;
}
