/* --------------------------------- stick.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for the joystick as a pointing device for UNIX/X11.
*/

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <linux/joystick.h>

#include "fly.h"

#define PO	p->opt
#define BSTICK	0x0001

static int	js[2] = {-1, -1};

static int FAR
jread (POINTER *p)
{
	struct JS_DATA_TYPE jsdata;
	int	which, px, py, x, y, ntimes;

	which = p->control->flags & BSTICK;
	if (js[which] < 0)
		return (1);

	px = PO[1];
	py = PO[3];

	x = y = 0;	/* Avoid compiler warning. */

	for (ntimes = 0; ntimes < PO[8]; ++ntimes) {
		if (read (js[which], (void *)&jsdata, JS_RETURN) != JS_RETURN)
			return (1);
		if (ntimes) {
			if (x > jsdata.x)
				x = jsdata.x;
			if (y > jsdata.y)
				y = jsdata.y;
		} else {
			x = jsdata.x;
			y = jsdata.y;
		}
	}
	if (jsdata.buttons & 1)
		++p->b[PO[4]];
	if (jsdata.buttons & 2)
		++p->b[PO[5]];
	p->a[px] = x;
	p->a[py] = y;

	if (p->c[px] == 0 || p->c[py] == 0)
		return (0);		/* calibrating */

#define	REF	100		/* expected full range */
#define	DULL	20		/* dull center */
#define	EDGE	10		/* dull edge */

#define	MDULL	(REF+DULL)
#define	FULL	(MDULL+EDGE)

	p->a[px] -= p->c[px];		/* center */
	p->a[px] *= -PO[0];		/* orientation */
	if (p->a[px] > p->c[px])	/* scale */
		p->a[px] = REF;
	else if (p->a[px] < -p->c[px])
		p->a[px] = -REF;
	else {
		p->a[px] = muldiv (p->a[px], FULL, p->c[px]);
		if      (p->a[px] > MDULL)
			p->a[px] = REF;
		else if (p->a[px] >= DULL)
			p->a[px] -= DULL;
		else if (p->a[px] < -MDULL)
			p->a[px] = -REF;
		else if (p->a[px] <= -DULL)
			p->a[px] += DULL;
		else
			p->a[px] = 0;
	}

	p->a[py] -= p->c[py];		/* center */
	p->a[py] *= PO[2];		/* orientation */
	if (p->a[py] > p->c[py])	/* scale */
		p->a[py] = REF;
	else if (p->a[py] < -p->c[py])
		p->a[py] = -REF;
	else {
		p->a[py] = muldiv (p->a[py], FULL, p->c[py]);
		if      (p->a[py] > MDULL)
			p->a[py] = REF;
		else if (p->a[py] >= DULL)
			p->a[py] -= DULL;
		else if (p->a[py] < -MDULL)
			p->a[py] = -REF;
		else if (p->a[py] <= -DULL)
			p->a[py] += DULL;
		else
			p->a[py] = 0;
	}
	return (0);
}

static int FAR
cal (POINTER *p)
/* Calibrate joy-stick. Paddle must be at center!
 *
*/
{
	p->c[PO[1]] = p->c[PO[3]] = 0;	/* indicate 'calibrating' */
	if (jread (p))
		return (1);
	p->c[PO[1]] = p->a[PO[1]];
	p->c[PO[3]] = p->a[PO[3]];

	p->a[PO[1]] = p->a[PO[3]] = 0;
	p->l[PO[1]] = p->l[PO[3]] = 0;
	if (p->c[PO[1]] == 0 || p->c[PO[3]] == 0)
		return (1);
	return (0);
}

static int FAR
init (POINTER *p)
{
	int	which;
	long	timelimit;

	which = p->control->flags & BSTICK;
	if ((js[which] = open (which ? "/dev/js1" : "/dev/js0", O_RDONLY)) < 0)
		return (1);

	timelimit = 5L;
	ioctl (js[which], JS_SET_TIMELIMIT, &timelimit);

	cal (p);

	return (0);
}

static void FAR
term (POINTER *p)
{
	int	which;

	which = p->control->flags & BSTICK;
	if (js[which] >= 0) {
		close (js[which]);
		js[which] = -1;
	}
}

struct PtrDriver PtrAstick = {
	"ASTICK",
	0,
	init,
	term,
	cal,
	cal,			/* center */
	jread,
	std_key
};

struct PtrDriver PtrBstick = {
	"BSTICK",
	BSTICK,
	init,
	term,
	cal,
	cal,			/* center */
	jread,
	std_key
};
